(function () {
    // --- Helpers ---
    function getCurrentUsername() {
        const path = window.location.pathname.split('/');
        return path[1] || 'default';
    }

    function hydrateProductOptions(username = getCurrentUsername(), productId) {
        const items = getCartItems(username);
        const item = items.find(i => i.id == productId);
        if (!item || !item.options) return;

        const el = document.querySelector('[x-data="productOptions()"]');
        if (!el) return;

        const alpineData = el.__x.$data;
        if (!alpineData) return;

        alpineData.selectedSize = item.options.size || null;
        alpineData.selectedColor = item.options.color || null;
        alpineData.selectedStyle = item.options.style || null;
    }

    function getCartKey(productId, username = getCurrentUsername()) {
        return `cart_${username}_${productId}`;
    }

    function getCartKeySubscription(productId, username = getCurrentUsername()) {
        return `cart_sub_${username}_${productId}`;
    }

    // --- Cart Storage Functions ---
    function getCartItems(username = getCurrentUsername()) {
        const items = [];
        for (let i = 0; i < localStorage.length; i++) {
            const key = localStorage.key(i);
            if (!key || !key.startsWith(`cart_${username}_`)) continue;
            try {
                const item = JSON.parse(localStorage.getItem(key));
                if (item && item.quantity > 0) items.push(item);
            } catch (e) {
                // ignore parse errors
            }
        }
        return items;
    }

    function getCartItemsSub(username = getCurrentUsername()) {
        // console.log('getCartItemsSub');
        const items = [];
        for (let i = 0; i < localStorage.length; i++) {
            const key = localStorage.key(i);
            if (!key || !key.startsWith(`cart_sub_${username}_`)) continue;
            try {
                const item = JSON.parse(localStorage.getItem(key));
                if (item && item.quantity > 0) items.push(item);
            } catch (e) {
                // ignore parse errors
            }
        }
        return items;
    }

    function saveToCart(
        productId,
        quantity,
        name = null,
        price = 0,
        image = null,
        username = getCurrentUsername(),
        options = {} // <-- new argument for size/color/style
    ) {
        const item = {
            id: productId,
            quantity: parseInt(quantity) || 0,
            name: name,
            price: parseFloat(price) || 0,
            image: image || null,
            username: username,
            options: {
                size: options.size || null,
                color: options.color || null,
                style: options.style || null
            }
        };

        if (item.quantity > 0) {
            localStorage.setItem(getCartKey(productId, username), JSON.stringify(item));
        } else {
            localStorage.removeItem(getCartKey(productId, username));
        }

        updateCartFooter(username);
        renderCartPage(username);
        syncCartToLivewire(username);
    }

    function getLatestSubscription(username) {
        const keys = Object.keys(localStorage);
        const subscriptionItems = [];

        keys.forEach(key => {
            if (key.startsWith(`cart_sub_${username}_`)) {
                const item = JSON.parse(localStorage.getItem(key));
                if (item?.subscription) {
                    subscriptionItems.push(item);
                }
            }
        });

        // ✅ most recent = last saved
        return subscriptionItems.pop() || null;
    }

    function deleteSubBeforeAdding(username) {
        Object.keys(localStorage).forEach((key) => {
            if (key.startsWith(`cart_sub_${username}_`)) {
                localStorage.removeItem(key);
            }
        });
    }

    // THIS WORKS
    function saveSubscriptionToCart(data) {

        const username = getCurrentUsername();

        //remove old subscriptions, only one product
        deleteSubBeforeAdding(username);

        const product = data.product ?? {};
        const tier = data; // ✅ tier values are at the root, not data.tier

        // console.log('data:', JSON.stringify(data));


        let item = {
            id: product.id,
            quantity: 1,
            name: product.product_name || tier.title || null,
            subscription: product.subscription || tier.subscription_link || null, // ✅ NO tier.subscription_link
            price: parseFloat(tier.price) || parseFloat(product.price) || 0,
            image: product.featured_image || null,
            description: tier.tier_description || product.description || null,
            username: username,
            tier_title: tier.title || null,
            tier_coin: tier.coin || product.coin || null,
        };

        localStorage.setItem(
            getCartKeySubscription(product.id, username),
            JSON.stringify(item)
        );

        updateCartFooter(username);
        renderCartPageSubs(username);
        syncCartToLivewireSub(username);
        updateSubscriptionButton(username);
    }

    function updateSubscriptionButton(username) {
        const latest = getLatestSubscription(username);
        const btn = document.getElementById("subscription-btn");
        const coin = document.getElementById("tier-coin");

        if (btn && latest?.subscription) {
            btn.href = latest.subscription;
        }

        if (coin && latest?.subscription) {
            coin.textContent = latest.tier_coin;
        }
    }

    function removeFromCart(productId, username = getCurrentUsername()) {
        localStorage.removeItem(getCartKey(productId, username));
        updateCartFooter(username);
        renderCartPage(username);
        renderCartPageSubs(username);
        syncCartToLivewire(username);
    }

    function removeFromCartSub(productId, username = getCurrentUsername()) {
        localStorage.removeItem(getCartKeySubscription(productId, username));
        // updateCartFooter(username);
        renderCartPageSubs(username);
        syncCartToLivewireSub(username);
    }

    function updateQuantity(productId, change, fromCartPage = false, username = getCurrentUsername()) {
        const inputId = fromCartPage
            ? `cart-input-${productId}`
            : `quantity-${productId}`;
        const input = document.getElementById(inputId);
        if (!input) return;

        let newValue = parseInt(input.value) + change;
        if (newValue < 0) newValue = 0;
        input.value = newValue;

        const name = input.getAttribute("data-name") || null;
        const price = parseFloat(input.getAttribute("data-price")) || 0;
        const image = input.getAttribute("data-image") || null;

        const options = {
            size: document.getElementById('selected-size')?.value,
            color: document.getElementById('selected-color')?.value,
            style: document.getElementById('selected-style')?.value
        };

        saveToCart(productId, newValue, name, price, image, username, options);
    }

    // --- UI Functions ---
    function updateCartFooter(username = getCurrentUsername()) {
        const items = getCartItems(username);
        const total = items.reduce((sum, item) => sum + (item.quantity * (item.price || 0)), 0);

        const footer = document.getElementById("cart-footer");
        const totalEl = document.getElementById("cart-total");

        if (totalEl) totalEl.textContent = total.toFixed(2);
        if (footer) {
            if (items.length > 0) footer.classList.remove("hidden");
            else footer.classList.add("hidden");
        }
    }

    function renderCartPage(username = getCurrentUsername()) {
        const container = document.getElementById("cart-items");
        const totalEl = document.getElementById("cart-page-total");
        const cartPage = document.getElementById("cart-page");
        const emptyCart = document.getElementById("empty-cart");

        if (!container) return;
        container.innerHTML = "";

        const items = getCartItems(username);

        if (items.length === 0) {
            if (cartPage) cartPage.classList.add("hidden");
            if (emptyCart) emptyCart.classList.remove("hidden");
            syncCartToLivewire(username);
            return;
        } else {
            if (cartPage) cartPage.classList.remove("hidden");
            if (emptyCart) emptyCart.classList.add("hidden");
        }

        let total = 0;
        items.forEach(item => {
            const subtotal = item.quantity * item.price;
            total += subtotal;

            container.innerHTML += `
<div class="flex items-center justify-between border-b">
    <div class="flex items-center gap-3 pb-5">
        ${item.image ? `<img src="/storage/${item.image}" class="w-16 h-16 rounded object-cover">` : ""}
        <div>
            <p class="font-bold text-gray-500 dark:text-white">
                ${(item.name && item.name.length > 30)
                    ? item.name.substring(0, 30) + "..."
                    : (item.name || ('Product #' + item.id))}
            </p>

            <p class="text-sm text-gray-500 dark:text-white">
                ${item.options.size ? 'Size: ' + item.options.size + ' | ' : ''}
                ${item.options.color ? 'Color: ' + item.options.color + ' | ' : ''}
                ${item.options.style ? 'Style: ' + item.options.style : ''}
            </p>

            <p class="text-sm text-gray-500 dark:text-white">Price: $${item.price.toFixed(2)}</p>
            <p class="font-bold text-gray-500 dark:text-white">$${subtotal.toFixed(2)}</p>
        </div>
    </div>
    <div class="flex flex-col items-center gap-1">
        <div class="flex items-center gap-1">
            <button onclick="updateQuantity('${item.id}', -1, true, '${username}')"
                    class="px-1.5 py-0.5 text-lg bg-gray-200 text-black rounded">-</button>
            <input type="number" id="cart-input-${item.id}"
                data-name="${item.name || ''}"
                data-image="${item.image || ''}"
                data-price="${item.price}"
                value="${item.quantity}"
                min="0"
                class="w-10 text-center text-lg text-black border rounded"
                onchange="saveToCart('${item.id}', this.value, this.getAttribute('data-name'), parseFloat(this.getAttribute('data-price')), this.getAttribute('data-image'), '${username}')">
            <button onclick="updateQuantity('${item.id}', 1, true, '${username}')"
                    class="px-1.5 py-0.5 text-lg bg-gray-200 text-black rounded">+</button>
        </div>
        <button onclick="removeFromCart('${item.id}', '${username}')"
                class="mt-2 px-2 py-0.5 bg-red-500 text-white rounded hover:bg-red-600 transition flex items-center gap-1 text-xs">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
                 stroke-width="1.5" stroke="currentColor" class="w-4 h-4">
                <path stroke-linecap="round" stroke-linejoin="round"
                      d="m14.74 9-.346 9m-4.788 0L9.26 9m9.968-3.21
                      c.342.052.682.107 1.022.166m-1.022-.165L18.16 19.673
                      a2.25 2.25 0 0 1-2.244 2.077H8.084a2.25 2.25 0 0 1-2.244-2.077L4.772
                      5.79m14.456 0a48.108 48.108 0 0 0-3.478-.397m-12
                      .562c.34-.059.68-.114 1.022-.165m0
                      0a48.11 48.11 0 0 1-3.478-.397m7.5
                      0v-.916c0-1.18-.91-2.164-2.09-2.201a51.964
                      51.964 0 0 0-3.32 0c-1.18.037-2.09
                      1.022-2.09 2.201v.916m7.5 0a48.667
                      48.667 0 0 0-7.5 0"/>
            </svg>
            Remove
        </button>
    </div>
</div>`;

        });

        if (totalEl) totalEl.textContent = total.toFixed(2);
        syncCartToLivewire(username);
    }

    function renderCartPageSubs(username = getCurrentUsername()) {

        const container = document.getElementById("cart-items-sub");
        const totalEl = document.getElementById("cart-page-total-sub");
        const cartPage = document.getElementById("cart-page-sub");
        const emptyCart = document.getElementById("empty-cart-sub");

        if (!container) return;
        container.innerHTML = "";

        const items = getCartItemsSub(username);
        // console.log(items);

        if (items.length === 0) {
            if (cartPage) cartPage.classList.add("hidden");
            if (emptyCart) emptyCart.classList.remove("hidden");
            syncCartToLivewireSub(username);
            return;
        } else {
            if (cartPage) cartPage.classList.remove("hidden");
            if (emptyCart) emptyCart.classList.add("hidden");
        }

        let total = 0;
        items.forEach(item => {
            // const subtotal = item.quantity * item.price;
            // total += subtotal;

            total = item.price;

            container.innerHTML += `
<div class="border-b">
    <div class="pb-5">
        ${item.image ? `<img src="/storage/${item.image}" class="w-full mx-auto rounded object-cover">` : ""}
        <div>
            <div class="text-base my-5 font-bold text-gray-500 dark:text-white text-center">
                ${(item.name && item.name.length > 30)
                    ? item.name.substring(0, 30) + "..."
                    : (item.name || ('Product #' + item.id))}
            </div>
            <p class="text-base font-bold text-gray-500 dark:text-white text-center">${item.tier_title ? item.tier_title : ''}</p>
                    <div class="text-sm prose my-5 text-gray-500 dark:text-white">${item.description}</div>
        </div>
        <div>
            <button onclick="removeFromCartSub('${item.id}','${username}')"
                    class="mt-2 py-2 bg-red-500 text-white rounded hover:bg-red-600 transition flex items-center w-full justify-center text-xs">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
                    stroke-width="1.5" stroke="currentColor" class="w-4 h-4">
                    <path stroke-linecap="round" stroke-linejoin="round"
                        d="m14.74 9-.346 9m-4.788 0L9.26 9m9.968-3.21
                        c.342.052.682.107 1.022.166m-1.022-.165L18.16 19.673
                        a2.25 2.25 0 0 1-2.244 2.077H8.084a2.25 2.25 0 0 1-2.244-2.077L4.772
                        5.79m14.456 0a48.108 48.108 0 0 0-3.478-.397m-12
                        .562c.34-.059.68-.114 1.022-.165m0
                        0a48.11 48.11 0 0 1-3.478-.397m7.5
                        0v-.916c0-1.18-.91-2.164-2.09-2.201a51.964
                        51.964 0 0 0-3.32 0c-1.18.037-2.09
                        1.022-2.09 2.201v.916m7.5 0a48.667
                        48.667 0 0 0-7.5 0"/>
                </svg>
                Remove
            </button>
         </div>
    </div>
</div>`;

        });

        if (totalEl) totalEl.textContent = total.toFixed(2);
        syncCartToLivewireSub(username);
    }

    function hydrateInputsFromCart(username = getCurrentUsername()) {
        const items = getCartItems(username);
        items.forEach(item => {
            const input = document.getElementById(`quantity-${item.id}`);
            if (input) input.value = item.quantity;

            const cartInput = document.getElementById(`cart-input-${item.id}`);
            if (cartInput) cartInput.value = item.quantity;
        });
    }

    // --- Livewire Sync ---
    function syncCartToLivewire(username = getCurrentUsername()) {
        const items = getCartItems(username);
        const hiddenInput = document.getElementById("cartData");
        if (hiddenInput) {
            hiddenInput.value = JSON.stringify(items);
            hiddenInput.dispatchEvent(new Event("input"));
        }
    }

    function syncCartToLivewireSub(username = getCurrentUsername()) {
        // console.log('hello');
        const items = getCartItemsSub(username);

        const hiddenInput = document.getElementById("cartData");
        if (hiddenInput) {
            hiddenInput.value = JSON.stringify(items);
            hiddenInput.dispatchEvent(new Event("input"));
        }
    }

    document.addEventListener('cart:clear', () => {
        const username = getCurrentUsername();
        for (let i = localStorage.length - 1; i >= 0; i--) {
            const key = localStorage.key(i);
            if (key && key.startsWith(`cart_${username}_`)) {
                localStorage.removeItem(key);
            }
        }
        renderCartPage(username);
        updateCartFooter(username);
    });

    document.addEventListener('cart:clear-sub', () => {
        const username = getCurrentUsername();
        for (let i = localStorage.length - 1; i >= 0; i--) {
            const key = localStorage.key(i);
            if (key && key.startsWith(`cart_sub_${username}_`)) {
                localStorage.removeItem(key);
            }
        }
        renderCartPageSubs(username);
        updateCartFooter(username);
    });

    // --- Expose globally ---
    window.getCartItems = getCartItems;
    window.saveToCart = saveToCart;
    window.saveSubscriptionToCart = saveSubscriptionToCart;
    window.removeFromCart = removeFromCart;
    window.removeFromCartSub = removeFromCartSub;
    window.updateQuantity = updateQuantity;
    window.updateCartFooter = updateCartFooter;
    window.renderCartPage = renderCartPage;
    window.renderCartPageSubs = renderCartPageSubs;
    window.hydrateProductOptions = hydrateProductOptions;

    // --- Init ---
    document.addEventListener("DOMContentLoaded", () => {
        const username = getCurrentUsername();
        updateCartFooter(username);
        renderCartPage(username);
        renderCartPageSubs(username);
        hydrateInputsFromCart(username);
        updateSubscriptionButton(username);

        // Hydrate product options for current product if Alpine exists
        const productId = document.querySelector('[x-data="productOptions()"]')?.dataset?.productId;
        if (productId) hydrateProductOptions(username, productId);
    });

    window.addEventListener("storage", () => {
        const username = getCurrentUsername();
        updateCartFooter(username);
        renderCartPage(username);
        renderCartPageSubs(username);
    });

    // document.addEventListener("livewire:update", () => {
    //     const username = getCurrentUsername();
    //     updateCartFooter(username);
    //     renderCartPage(username);
    //     hydrateInputsFromCart(username);

    //     const productId = document.querySelector('[x-data="productOptions()"]')?.dataset?.productId;
    //     if (productId) hydrateProductOptions(username, productId);
    // });

    document.addEventListener("livewire:update", () => {
        const username = getCurrentUsername();

        // prevent empty-cart flicker during checkout/receipt upload
        const hiddenInput = document.getElementById("cartData");
        const hasCartData = hiddenInput && hiddenInput.value && hiddenInput.value.trim() !== "[]";

        if (hasCartData) {
            updateCartFooter(username);
            hydrateInputsFromCart(username);
            return; // skip re-render to keep the checkout visible
        }

        // only render when actual cart view (not checkout)
        updateCartFooter(username);
        renderCartPage(username);
        renderCartPageSubs(username);
        hydrateInputsFromCart(username);
    });

})();
