<?php

namespace App\Services;

use App\Models\User;
use App\Models\Referral;
use App\Models\SubCard;
use App\Models\Commissions;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Log;

class CommissionService
{
    public function __construct(
        protected CardDefaultsSetting $settings
    ) {}

    public function createCommission(User $user, ?string $invoiceId = null, ?float $amountPaid = null): ?Commissions
    {
        $refer = Referral::where('referred_id', $user->id)->first();

        if (! $refer) {
            Log::info("No referral found for user {$user->id}");
            return null;
        }

        // Resolve referrer
        if ($refer->referrer_type === 'main') {
            $referrerId = $refer->referrer_id_main;
            $referrer   = User::find($referrerId);
        } else {
            $referrerId = $refer->referrer_id_sub;
            $subcard    = SubCard::find($referrerId);

            if (! $subcard) {
                Log::warning("SubCard not found: {$referrerId}");
                return null;
            }

            $referrer = User::find($subcard->user_id);
        }

        if (! $referrer) {
            Log::warning("Referrer not found for user: {$user->id}");
            return null;
        }

        $partnershipLevel = $referrer->partnership_level;

        // Commission rate mapping
        $rate = match ($partnershipLevel) {
            'senior' => 0.50,
            'junior' => 0.25,
            'leader' => 0.20,
            default  => 0.10,
        };

        // If amountPaid is not passed, compute from settings based on user_type & billing cycle
        if (is_null($amountPaid)) {
            $amountPaid = $this->resolvePrice($user);
        }

        $commissionData = [
            'referred_id'     => $refer->referred_id,
            'commission_rate' => $rate * 100,
            'amount'          => $amountPaid * $rate,
        ];

        if ($invoiceId && is_string($invoiceId)) {
            $commissionData['stripe_invoice_id'] = $invoiceId;
        }

        if ($refer->referrer_type === 'main') {
            $commissionData['user_id'] = $referrer->id;
            $commissionData['page_id'] = null;
        } else {
            $commissionData['user_id'] = null;
            $commissionData['page_id'] = $referrerId;
        }

        return Commissions::create($commissionData);
    }

    private function resolvePrice(User $user): float
    {
        $plan = strtolower($user->user_type); // "basic", "gold", "platinum"

        // normalize access (Month, Year, Subscription, Trial, Lifetime)
        $billing = match (strtolower($user->access)) {
            'month' => 'monthly',
            'year'  => 'yearly',
            default => 'monthly', // fallback
        };

        return match ("{$plan}_{$billing}") {
            'basic_monthly'     => $this->settings->price_basic_monthly,
            'basic_yearly'      => $this->settings->price_basic_yearly,
            'gold_monthly'      => $this->settings->price_gold_monthly,
            'gold_yearly'       => $this->settings->price_gold_yearly,
            'platinum_monthly'  => $this->settings->price_platinum_monthly,
            'platinum_yearly'   => $this->settings->price_platinum_yearly,
            default             => 0,
        };
    }
}
