<?php

namespace App\Observers;

use App\Models\TrustedNetwork;
use App\Models\User;
use App\Models\SubCard;
use Filament\Notifications\Notification;
use App\Notifications\UserNotification;
use Filament\Notifications\Actions\Action;
use Illuminate\Support\Facades\Auth;
use App\Mail\OrderMail;
use Illuminate\Support\Facades\Mail;
use App\Settings\CardDefaultsSetting;

class TrustedNetworkObserver
{
    public function created(TrustedNetwork $trusted_network): void
    {
        if (Auth::check()) {
            // Owner should not notify
            if (Auth::id() === (int) $trusted_network->user_id) {
                return;
            }

            // Admin should not notify
            if (Auth::user()->is_admin) {
                return;
            }
        }

        $member = User::find($trusted_network->member_id);

        $recipient = User::where('is_admin', true)->get();

        // Determine owner (User or SubCard)
        $owner = User::find($trusted_network->user_id)
            ?? SubCard::find($trusted_network->page_id);

        // Determine owner type
        if ($owner instanceof User) {
            $owner_type = 'main';
            // $owner_id = $owner->id;
        } else {
            $owner_type = 'sub';
            // $owner_id = $owner->id;
        }

        // Determine member (User or SubCard)
        if ($trusted_network->card_type == 'main') {
            $user = User::find($trusted_network->member_id);
        } else {
            $user = SubCard::find($trusted_network->member_id);
        }

        // Build the ACTION dynamically
        $action = ($owner_type === 'main')
            ? Action::make('view')
            ->url(route('filament.admin.resources.clients.edit', [
                'record' => $trusted_network->user_id,
                'activeRelationManager' => 17 //17 is Trusted Network Tab of Client
            ]))
            : Action::make('view')
            ->url(route('filament.admin.resources.sub-cards.edit', [
                'record' => $trusted_network->page_id,
                'activeRelationManager' => 16 //16 is Trusted Network Tab of SubCard
            ]));

        if ($owner_type === 'main') {
            $recordUrl = route('filament.admin.resources.clients.edit', [
                'record' => $trusted_network->user_id,
                'activeRelationManager' => 17 //17 is Trusted Network Tab of Client
            ]);
        } else {
            $recordUrl = route('filament.admin.resources.sub-cards.edit', [
                'record' => $trusted_network->page_id,
                'activeRelationManager' => 16 //16 is Trusted Network Tab of SubCard
            ]);
        }

        // Send notification
        Notification::make()
            ->title('Trusted Network: Request to Join')
            ->body("{$user->firstname} {$user->lastname} requested to join {$owner->firstname} {$owner->lastname}'s network.")
            ->actions([$action])
            ->sendToDatabase($recipient);

        $default = app(CardDefaultsSetting::class);

        // if turned on, admin email(s) will receive email notification
        if ($default->admin_email_notification) {

            $emails = collect(explode(',', $default->admin_email))
                ->map(fn($email) => trim($email))
                ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));

            Mail::to($emails)
                ->queue(
                    new OrderMail(
                        "Hey admin!<br><br>
                {$user->firstname} {$user->lastname} is requesting to join {$owner->firstname} {$owner->lastname}'s network. <br><br>
                <a href='{$recordUrl}' target='_blank'>View request</a>
                ",
                        'Trusted Network: Request to Join',
                        $default->sender_email
                    )
                );
        }

        // -------------------------------
        // USER (LIVEWIRE) NOTIFICATION
        // -------------------------------
        if ($owner instanceof User && ! $owner->is_admin) {
            $owner->notify(
                new UserNotification(
                    title: 'Trusted Network Request',
                    message: "<span class='font-semibold'>{$user->firstname} {$user->lastname}</span> requested to join your trusted network.",
                    url: route('dashboard.trustednetwork')
                )
            );
        }
    }
}
