<?php

namespace App\Observers;

use App\Models\FormSubmission;
use App\Models\LeadCaptureForm;
use App\Models\User;
use App\Models\SubCard;
use Filament\Notifications\Notification;
use App\Mail\OrderMail;
use Illuminate\Support\Facades\Mail;
use App\Settings\CardDefaultsSetting;
use App\Notifications\UserNotification;
use App\Services\TwilioService;

class FormSubmissionObserver
{
    public function created(FormSubmission $form): void
    {
        $recipient = User::where('is_admin', true)->get();
        $form_info = LeadCaptureForm::where('id', $form->form_id)->first();

        if (!empty($form_info->page_id)) {
            \Log::info("Page Id: {$form_info->page_id}");
            // Subcard
            $owner = SubCard::find($form_info->page_id);
            if (! $owner) {
                \Log::warning("SubCard not found: {$form_info->page_id}");
                return;
            }

            // Notify the subcard itself
            $owner->notify(
                new UserNotification(
                    title: "Form Submission: {$form_info->title}",
                    message: "A new form submission from <span class='font-semibold'>{$form->firstname} {$form->lastname}</span> has been received. Please review the information provided.",
                    url: route('submission.show', $form->id),
                )
            );
        } elseif (!empty($form_info->user_id)) {
            \Log::info("User Id: {$form_info->user_id}");
            // Main card user
            $owner = User::find($form_info->user_id);
            if (! $owner) return;

            if ($owner->is_admin) {
                return;
            }

            $owner->notify(
                new UserNotification(
                    title: "Form Submission: {$form_info->title}",
                    message: "A new form submission from <span class='font-semibold'>{$form->firstname} {$form->lastname}</span> has been received. Please review the information provided.",
                    url: route('submission.show', $form->id),
                )
            );
        }

        $recordUrl = route(
            'filament.admin.resources.form-submissions.view',
            ['record' => $form->id]
        );

        Notification::make()
            ->title("Form submission: {$form_info->title}")
            ->body("A new form submission from {$form->firstname} {$form->lastname} has been received. Please review the information provided.")
            ->actions([
                \Filament\Notifications\Actions\Action::make('view')
                    ->url($recordUrl)

            ])
            ->sendToDatabase($recipient);

        $default = app(CardDefaultsSetting::class);
        $twilio = new TwilioService();

        $message = "Hey admin!\nA new form submission from {$form->firstname} {$form->lastname} has been received. Please review the information provided.\n"
            . "\n{$recordUrl}";

        // if turned on, admin will receive sms notification
        if ($default->admin_sms_notification && $default->admin_sms_number) {

            $numbers = collect(explode(',', $default->admin_sms_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendSms($number, $message);
                } catch (\Exception $e) {
                    \Log::error("SMS failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin will receive whatsapp notification
        if ($default->admin_whatsapp_notification && $default->admin_whatsapp_number) {

            $numbers = collect(explode(',', $default->admin_whatsapp_number))
                ->map(fn($n) => trim($n))
                ->filter();

            foreach ($numbers as $number) {
                try {
                    $twilio->sendWhatsapp($number, $message);
                } catch (\Exception $e) {
                    \Log::error("WhatsApp failed: " . $e->getMessage());
                }
            }
        }

        // if turned on, admin email(s) will receive email notification
        if ($default->admin_email_notification) {

            $emails = collect(explode(',', $default->admin_email))
                ->map(fn($email) => trim($email))
                ->filter(fn($email) => filter_var($email, FILTER_VALIDATE_EMAIL));

            Mail::to($emails)
                ->queue(
                    new OrderMail(
                        "Hey admin!<br><br>
                A new form submission from <strong>{$form->firstname} {$form->lastname}</strong> has been received. Please review the information provided.<br><br>
                <a href='{$recordUrl}' target='_blank'>{$recordUrl}</a>
                ",
                        "Form submission: {$form_info->title}",
                        $form->sender_email
                    )
                );
        }
    }
}
