<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\User;
use App\Models\SubCard;
use App\Models\SaleVideo;
use App\Models\SaleImage;
use Spatie\EloquentSortable\Sortable;
use Spatie\EloquentSortable\SortableTrait;
use Illuminate\Support\Str;

class Sale extends Model implements Sortable
{
    use SortableTrait;

    protected $fillable = [
        'user_id',
        'page_id',
        'slug',
        'title',
        'description',
        'order',
        'is_active',
        'thumbnail',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function subcard()
    {
        return $this->belongsTo(SubCard::class, 'page_id');
    }

    public function sale_videos()
    {
        return $this->hasMany(SaleVideo::class, 'sales_id');
    }

    public function sale_images()
    {
        return $this->hasMany(SaleImage::class, 'sales_id');
    }

    protected static function booted()
    {
        static::creating(function ($sale) {

            if (empty($sale->slug)) {
                $baseSlug = Str::slug($sale->title);
                $slug = $baseSlug;
                $count = 1;

                while (static::where('slug', $slug)->exists()) {
                    $slug = $baseSlug . '-' . $count++;
                }

                $sale->slug = $slug;
            }
        });

        static::saving(function ($sale) {

            if (empty($sale->slug) || $sale->isDirty('title')) {
                $baseSlug = Str::slug($sale->title);
                $slug = $baseSlug;
                $count = 1;

                while (static::where('slug', $slug)
                    ->where('id', '!=', $sale->id)
                    ->exists()
                ) {
                    $slug = $baseSlug . '-' . $count++;
                }

                $sale->slug = $slug;
            }
        });

        static::updating(function ($sale) {
            if ($sale->isDirty('thumbnail') && $sale->getOriginal('thumbnail')) {
                $path = 'app/public/' . $sale->getOriginal('thumbnail');
                @unlink(storage_path($path));
            }
        });

        static::deleting(function ($sale) {

            // Delete Sales Page Thumbnail
            $path = 'app/public/' . $sale->thumbnail;
            if ($sale->thumbnail) {
                @unlink(storage_path($path));
            }

            foreach ($sale->sale_images as $image) {
                $path = storage_path('app/public/' . $image->image_path);

                if (file_exists($path)) {
                    @unlink($path);
                }

                $image->delete(); // remove db record
            }
        });
    }
}
