<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Spatie\EloquentSortable\Sortable;
use Spatie\EloquentSortable\SortableTrait;
use App\Models\User;
use App\Models\SubCard;
use Illuminate\Support\Str;
use App\Helpers\SavingLinkHelpers;

class Event extends Model implements Sortable
{
    use SortableTrait;

    protected $table = 'events';
    protected $fillable = [
        'title',
        'slug',
        'user_id',
        'page_id',
        'button_title',
        'button_link',
        'event_poster',
        'images',
        'description',
        'location',
        'start_datetime',
        'end_datetime',
        'timezone',
        'capacity',
        'order',
        'event_type',
        'event_video_link',
        'poster_active',
        'partner_link',
        'shop_link',
        'title_active',
        'contact_active',
        'shop_active',
        'ad_active',
        'contact_settings',
        'optional_cta'
    ];

    protected $casts = [
        'optional_cta'      => 'array',
        'event_video_link'  => 'array',
        'images'            => 'array',
        'contact_settings'  => 'array',
        'start_datetime'    => 'date',
    ];

    public function getEventVideoLinkAttribute($value)
    {
        if ($this->exists) {
            // Editing → decode JSON into array
            return is_string($value) ? json_decode($value, true) : $value;
        }

        // Creating → return raw so your CreateAction logic works
        return $value;
    }

    public function setEventVideoLinkAttribute($value)
    {
        if (is_array($value)) {
            if (isset($value[0][0]['link'])) {
                $value = array_map(fn($item) => $item[0]['link'], $value);
            } elseif (isset($value[0]['link'])) {
                $value = array_map(fn($item) => $item['link'], $value);
            }
            $this->attributes['event_video_link'] = json_encode($value);
        } elseif (is_string($value)) {
            $this->attributes['event_video_link'] = $value;
        } else {
            $this->attributes['event_video_link'] = null;
        }
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function subcard()
    {
        return $this->belongsTo(SubCard::class, 'page_id');
    }

    protected static function booted()
    {
        static::creating(function ($event) {

            if (empty($event->slug)) {
                $baseSlug = Str::slug($event->title);
                $slug = $baseSlug;
                $count = 1;

                while (static::where('slug', $slug)->exists()) {
                    $slug = $baseSlug . '-' . $count++;
                }

                $event->slug = $slug;
            }

            if ($event->event_video_link && is_array($event->event_video_link)) {
                $formatted = [];

                foreach ($event->event_video_link as $item) {
                    // Support both formats: ['link' => '...'] or just '...'
                    $link = is_array($item) ? ($item['link'] ?? null) : $item;

                    if (!empty($link)) {
                        $formatted[] = SavingLinkHelpers::saveEventVideoLink($link);
                    }
                }

                // Store as flat array of strings
                $event->event_video_link = $formatted;
            }

            if ($event['partner_link']) {
                static::handleLinkFormatting($event);
            }

            if ($event['shop_link']) {
                static::handleLinkFormatting($event);
            }
        });

        static::saving(function ($event) {

            // if ($event->event_video_link && is_array($event->event_video_link)) {
            //     $event->event_video_link = static::formatVideoLinks($event->event_video_link);
            // }

            if (empty($event->slug) || $event->isDirty('title')) {
                $baseSlug = Str::slug($event->title);
                $slug = $baseSlug;
                $count = 1;

                while (static::where('slug', $slug)
                    ->where('id', '!=', $event->id)
                    ->exists()
                ) {
                    $slug = $baseSlug . '-' . $count++;
                }

                $event->slug = $slug;
            }

            // if ($event->event_video_link && is_array($event->event_video_link)) {
            //     $formatted = [];
            //     foreach ($event->event_video_link as $link) {
            //         if (!empty($link)) {
            //             $formatted[] = SavingLinkHelpers::saveEventVideoLink($link);
            //         }
            //     }
            //     $event->event_video_link = $formatted; // still an array, Laravel casts to JSON
            // }

            if ($event->event_video_link && is_array($event->event_video_link)) {
                $formatted = [];

                foreach ($event->event_video_link as $item) {
                    // Support both formats: ['link' => '...'] or just '...'
                    $link = is_array($item) ? ($item['link'] ?? null) : $item;

                    if (!empty($link)) {
                        $formatted[] = SavingLinkHelpers::saveEventVideoLink($link);
                    }
                }

                // Store as flat array of strings
                $event->event_video_link = $formatted;
            }


            if ($event['partner_link']) {
                static::handleLinkFormatting($event);
            }

            if ($event['shop_link']) {
                static::handleLinkFormatting($event);
            }
        });

        static::updating(function ($event) {
            if ($event->isDirty('event_poster') && $event->getOriginal('event_poster')) {
                $path = 'app/public/' . $event->getOriginal('event_poster');
                @unlink(storage_path($path));
            }
        });

        static::deleting(function ($event) {
            $path = 'app/public/' . $event->event_poster;
            if ($event->event_poster) {
                @unlink(storage_path($path));
            }
        });
    }

    protected static function handleLinkFormatting($event)
    // protected static function handleLinkFormatting(array $event): array
    {
        // if (!empty($event['event_video_link']) && is_array($event['event_video_link'])) {
        //     $event['event_video_link'] = SavingLinkHelpers::saveEventVideoLink($event['event_video_link']);
        // }

        if (!empty($event['event_video_link']) && is_array($event['event_video_link'])) {
            $event['event_video_link'] = SavingLinkHelpers::saveEventVideoLink($event['event_video_link']);
        }

        // partner_link
        if (isset($event['partner_link'])) {
            try {
                $event['partner_link'] = SavingLinkHelpers::saveSecureLink($event['partner_link']);
            } catch (\InvalidArgumentException $e) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'partner_link' => 'Invalid input. Please enter a valid URL/link.',
                ]);
            }
        }

        // shop_link
        if (isset($event['shop_link'])) {
            try {
                $event['shop_link'] = SavingLinkHelpers::saveSecureLink($event['shop_link']);
            } catch (\InvalidArgumentException $e) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'shop_link' => 'Invalid input. Please enter a valid URL/link.',
                ]);
            }
        }

        return $event;
    }
}
