<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Video;
use App\Models\User;
use App\Models\SubCard;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Validation\ValidationException;

class VideoEdit extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $old_thumbnail;
    public $app_name;

    public $video, $title, $video_type, $link, $button_title, $button_link, $video_thumbnail;

    public $videoId, $video_owned_by_user;
    public $condition, $page_id;

    protected $rules = [
        'video_thumbnail'       => 'nullable|string',
        'title'                 => 'required|string',
        'video_type'            => 'required|string',
        'link'                  => 'required|url',
        'button_title'          => 'nullable|string',
        'button_link'           => 'nullable|url',
    ];

    #[On('videoThumbnail')]
    public function setVideoThumbnail($videoThumbnailUrl)
    {
        $this->video_thumbnail = new TemporaryUploadedFile($videoThumbnailUrl, config('filesystems.default'));
    }

    public function updateVideo()
    {
        try {
            $this->validate();

            if (auth()->user()) {
                $updateData = [
                    'title' => $this->title,
                    'video_type' => $this->video_type,
                    'link' => $this->link,
                    'button_title' => $this->button_title,
                    'button_link' => $this->button_link,
                ];

                // if ($this->video_thumbnail) {
                if ($this->video_thumbnail instanceof TemporaryUploadedFile && $this->video_thumbnail->isValid()) {
                    // Delete old thumbnail if exists
                    if ($this->old_thumbnail && Storage::disk('public')->exists($this->old_thumbnail)) {
                        Storage::disk('public')->delete($this->old_thumbnail);
                    }

                    // Store new thumbnail and save path
                    $updateData['thumbnail'] = $this->video_thumbnail->storePublicly('video_thumbnails', 'public');
                }

                // Update the user's video
                auth()->user()->videos()->where('id', $this->videoId)->update($updateData);

                session()->flash('success', 'Video successfully updated!');
            } else {
                session()->flash('error', 'Unauthorized action.');
            }
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }

        return $this->redirect('/dashboard/video', navigate: true);
    }

    public function viewVideo($id)
    {
        $this->video = Video::where('user_id', auth()->user()->id)
            ->where('id', $id)
            ->select('id', 'title', 'link', 'video_type', 'button_title', 'button_link', 'thumbnail')
            ->first();

        if (!$this->video) {
            abort(403, 'Access denied');
        }
    }

    public function mount($id)
    {
        $this->videoId = $id;

        $checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        if ($checkifactive > 0) {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        } else {
            $this->page_id = SubCard::where('user_id', auth()->user()->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();
            $this->condition = ['page_id' => $this->page_id];
        }

        $this->video_owned_by_user = auth()->user()->videos()
            ->where('id', $this->videoId)
            ->where($this->condition)
            ->count();

        $this->viewVideo($this->videoId);

        if ($this->video_owned_by_user > 0) {
            $this->old_thumbnail = $this->video->thumbnail;
            $this->title = $this->video->title;
            $this->video_type = $this->video->video_type;
            $this->link = $this->video->link;
            $this->button_title = $this->video->button_title;
            $this->button_link = $this->video->button_link;
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        // dd($this->old_thumbnail);
    }

    public function render()
    {
        $this->authorize('update', Video::class);
        return view('pages.video-edit')->title('Update Video | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
