<?php

namespace App\Livewire;

use Livewire\Component;
use App\Settings\CardDefaultsSetting;
use App\Models\UserSettings;
use App\Models\User;
use App\Models\SubCard;
use App\Models\Button;
use App\Models\OrderRecord;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\OrderMail;

class SalesPageCart extends Component
{
    use WithFileUploads;

    public $site_url;
    public $site_name;
    public $thumbnail;
    public $favicon;
    public $shop_thumbnail;
    public $app_name;

    public $color_settings_profile_pic_border;
    public $color_settings_custom_button;
    public $color_settings_custom_button_text;
    public $color_settings_card_light;
    public $color_settings_card_dark;
    public $color_settings_text_light;
    public $color_settings_text_dark;

    public $checkMain, $checkSub, $user, $condition, $page_id;
    public $payments;
    public $cartData;

    public $preferred_contact = 'Text';
    public $username, $firstname, $lastname, $phone, $receipt, $shipping_address, $special_note, $sender_email;
    public $products_item = [];

    public function mount($username)
    {
        $this->username = $username;
        $this->checkMain = User::where('username', $username)->count();
        $this->checkSub = SubCard::where('username', $username)->count();

        if ($this->checkMain > 0) {
            $this->user = User::where('username', $username)->where('is_active', true)->first();
            $this->condition = ['user_id' => $this->user->id, 'page_id' => NULL];
        }

        if ($this->checkSub > 0) {
            $this->page_id = SubCard::where('username', $username)->value('id');
            $this->user = SubCard::where('username', $username)->where('is_active', true)->first();
            $this->condition = ['page_id' => $this->page_id];
        }

        // Load default and user settings...
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->favicon = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->color_settings_custom_button = UserSettings::where('setting_key', 'color_settings_custom_button')
            ->where($this->condition)
            ->value('setting_value') ?? '#facc15';

        $this->color_settings_custom_button_text = UserSettings::where('setting_key', 'color_settings_custom_button_text')
            ->where($this->condition)
            ->value('setting_value') ?? '#000';

        $this->color_settings_profile_pic_border = UserSettings::where('setting_key', 'color_settings_profile_pic_border')
            ->where($this->condition)
            ->value('setting_value') ?? '#ca8a04';

        $this->color_settings_card_light = UserSettings::where('setting_key', 'color_settings_card_light')
            ->where($this->condition)
            ->value('setting_value') ?? '#fff';

        $this->color_settings_card_dark = UserSettings::where('setting_key', 'color_settings_card_dark')
            ->where($this->condition)
            ->value('setting_value') ?? '#18181b';

        $this->color_settings_text_light = UserSettings::where('setting_key', 'color_settings_text_light')
            ->where($this->condition)
            ->value('setting_value') ?? '#1e293b';

        $this->color_settings_text_dark = UserSettings::where('setting_key', 'color_settings_text_dark')
            ->where($this->condition)
            ->value('setting_value') ?? '#cbd5e1';

        // Shop thumbnail
        $this->shop_thumbnail = UserSettings::where('setting_key', 'shop_thumbnail')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->payments = Button::where('is_active', true)
            ->where($this->condition)
            ->where('view', 'payments')
            ->orderBy('order', 'asc')
            ->get();

        if ($this->payments->isEmpty()) {
            $this->payments = 0;
        }

        if ($this->shop_thumbnail) {
            // Use shop thumbnail if available
            $this->thumbnail = $this->shop_thumbnail;
            $this->favicon = $this->shop_thumbnail;
        } else if ($this->user->thumbnail) {
            // User has a custom thumbnail
            $this->thumbnail = $this->user->thumbnail;
            $this->favicon = $this->user->thumbnail;
        } else if ($this->user->profile_pic) {
            // Use user's profile picture
            $this->thumbnail = $this->user->profile_pic;
            $this->favicon = $this->user->profile_pic;
        } else {
            // Default site thumbnail fallback
            $this->thumbnail = $this->thumbnail ?? null;
            $this->favicon = $this->thumbnail;
        }
    }

    public function checkoutForm()
    {
        $cart = json_decode($this->cartData, true);

        $totalAmount = collect($cart)->reduce(function ($carry, $item) {
            return $carry + ($item['price'] * $item['quantity']);
        }, 0);

        $itemsString = collect($cart)->map(function ($item) {
            $details = [];
            if (!empty($item['options']['size'])) $details[] = "Size: {$item['options']['size']}";
            if (!empty($item['options']['color'])) $details[] = "Color: {$item['options']['color']}";
            if (!empty($item['options']['style'])) $details[] = "Style: {$item['options']['style']}";
            $detailsString = count($details) ? ' (' . implode(', ', $details) . ')' : '';
            $total = $item['price'] * $item['quantity'];
            return "{$item['name']}{$detailsString} ({$item['quantity']}x - $" . number_format($total, 2) . ")";
        })->implode(', ');

        $validated = $this->validate([
            'firstname'             => 'required|string|max:50',
            'lastname'              => 'required|string|max:50',
            'special_note'          => 'nullable|string|max:100',
            'shipping_address'      => 'required|string',
            'sender_email'          => 'required|email',
            'phone'                 => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
            'preferred_contact'     => 'required|in:Text,WhatsApp,Email',
            'receipt'               => 'required|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ], [
            'firstname.required'    => 'This field is required.',
            'lastname.required'     => 'This field is required.',
            'phone.regex'           => 'Please enter a valid phone number.',
            'receipt.max'           => 'Receipt must be less than 2MB.',
        ]);

        $receipt_path = $this->receipt->store('receipts', 'public');
        $receipt_url = Storage::url($receipt_path);

        // Save ALL items to database
        foreach ($cart as $item) {
            $this->products_item[] = [
                'product_id'        => $item['id'],
                'price'             => $item['price'],
                'quantity'          => $item['quantity'],
                'options'           => $item['options'] ?? [],
            ];
        }

        $noteHtml = !empty($validated['special_note'])
            ? "<br><strong>Special Note:</strong> {$validated['special_note']}<br>"
            : '';

        $shippingAddressHtml = !empty($validated['shipping_address'])
            ? "<br><strong>Shipping Address:</strong> {$validated['shipping_address']}<br><br>"
            : '';

        $noteText = !empty($validated['special_note'])
            ? "%0A%0ASpecial Note: {$validated['special_note']}"
            : '';

        $shippingAddressText = !empty($validated['shipping_address'])
            ? "%0A%0AShipping Address: {$validated['shipping_address']}%0A%0A"
            : '';

        $body = "Hey {$this->user->firstname}, this is {$validated['firstname']} {$validated['lastname']}.<br><br>"
            . "I’d like to purchase {$itemsString}.<br><br>"
            . "<strong>Total: $" . number_format($totalAmount, 2) . "</strong><br>"
            . $noteHtml
            . $shippingAddressHtml
            . "I have attached the payment receipt.<br><br>"
            . "Download Receipt: <a href='" . url($receipt_url) . "' target='_blank'>" . url($receipt_url) . "</a><br><br>"
            . "You can reach me at: <a href='mailto:{$validated['sender_email']}' target='_blank'>{$validated['sender_email']}</a>.<br><br>"
            . "It’s great connecting with you!<br><br>"
            . "<a href='{$this->site_url}{$this->user->username}' target='_blank'>{$this->site_url}{$this->user->username}/</a>";

        $message_to_buyer = "Hey {$validated['firstname']} {$validated['lastname']}!<br><br>"
            . "We received your order!<br>{$itemsString}.<br><br>"
            . "<strong>Total: $" . number_format($totalAmount, 2) . "</strong><br>"
            . $noteHtml
            . $shippingAddressHtml
            . "<strong>Attached receipt</strong><br>"
            . "Download Receipt: <a href='" . url($receipt_url) . "' target='_blank'>" . url($receipt_url) . "</a><br><br>"
            . "It’s great connecting with you!<br><br>"
            . "<a href='{$this->site_url}{$this->user->username}' target='_blank'>{$this->site_url}{$this->user->username}/</a>";

        $itemsStringEncoded = rawurlencode($itemsString);

        $message = "Hey {$this->user->firstname}, this is {$validated['firstname']} {$validated['lastname']}.%0A%0A"
            . "I’d like to purchase {$itemsStringEncoded}.%0A%0A"
            . "Total: $" . number_format($totalAmount, 2) . "%0A"
            . $noteText
            . $shippingAddressText
            . "I have attached the payment receipt.%0A%0A"
            . "Download Receipt: " . url($receipt_url)
            . "%0A%0AYou can reach me at: {$validated['sender_email']}.%0A%0A"
            . "It’s great connecting with you!%0A%0A"
            . "{$this->site_url}{$this->user->username}/";

        $smsUrl = "sms:{$this->user->phone}?&body={$message}";
        $whatsappUrl = "https://wa.me/{$this->user->phone}?text={$message}";

        try {
            OrderRecord::create(array_merge(
                $this->condition,
                [
                    'products'         => $this->products_item,
                    'price'            => $totalAmount,
                    'receipt'          => $receipt_path,
                    'firstname'        => $validated['firstname'],
                    'lastname'         => $validated['lastname'],
                    'email'            => $validated['sender_email'],
                    'phone'            => $validated['phone'],
                    'special_note'     => $validated['special_note'],
                    'shipping_address' => $validated['shipping_address'],
                ]
            ));

            if ($validated['preferred_contact'] === 'Text') {
                $this->dispatch('cart:clear');
                return redirect()->to($smsUrl);
            } elseif ($validated['preferred_contact'] === 'WhatsApp') {
                $this->dispatch('cart:clear');
                return redirect()->to($whatsappUrl);
            } else {
                // // Mail to seller
                // Mail::html($body, function ($mail) use ($validated) {
                //     $mail->to($this->user->email)
                //         ->from(config('mail.from.address'), config('mail.from.name'))
                //         ->replyTo($validated['sender_email'])
                //         ->subject('Order');
                // });

                // // Mail to buyer
                // $user = $this->user;

                // // Mail to buyer
                // Mail::html($message_to_buyer, function ($mail) use ($validated, $user) {
                //     $mail->to($validated['sender_email'])
                //         ->from(config('mail.from.address'), config('mail.from.name'))
                //         ->replyTo($user->email)
                //         ->subject('Order Received!');
                // });

                // Mail to seller
                Mail::to($this->user->email)
                    ->queue(new OrderMail($body, 'Order', $validated['sender_email']));

                // Mail to buyer
                Mail::to($validated['sender_email'])
                    ->queue(new OrderMail($message_to_buyer, 'Order Received!', $this->user->email));

                $this->dispatch('cart:clear');
                session()->flash('success', "Email sent to {$this->user->email}");
                return redirect(request()->header('Referer'));
            }
        } catch (\Exception $e) {
            // dd($e->getMessage());
            // Log::error('SalesPageCart email error: ' . $e->getMessage(), [
            //     'user_id' => $this->user->id ?? null,
            //     'username' => $this->username ?? null,
            //     'cartData' => $this->cartData ?? null,
            // ]);

            session()->flash('error', "Failed to send email. Please try again later.");
        }
    }

    public function render()
    {
        return view('livewire.sales-page-cart')->title('Cart')->layoutData([
            'favicon' => $this->thumbnail,
            'site_url' => $this->site_url,
            'thumbnail' => $this->thumbnail,
            'title' => 'Cart',
            'color_settings_card_light' => $this->color_settings_card_light,
            'color_settings_card_dark' => $this->color_settings_card_dark,
            'color_settings_text_light' => $this->color_settings_text_light,
            'color_settings_text_dark' => $this->color_settings_text_dark,
            'color_settings_profile_pic_border' => $this->color_settings_profile_pic_border,
            'color_settings_custom_button_text' => $this->color_settings_custom_button_text,
            'color_settings_custom_button' => $this->color_settings_custom_button,
            'app_name' => $this->app_name,
        ]);
    }
}
