<?php

namespace App\Livewire;

use App\Models\Referral;
use App\Models\SubCard;
use App\Models\User;
use Livewire\Component;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\DB;
use Livewire\WithPagination;
use Illuminate\Pagination\LengthAwarePaginator;
use Carbon\Carbon;

class ReferralPage extends Component
{
    use WithPagination;
    protected $paginationTheme = 'tailwind';

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $affiliate_functionality;

    public $user_type;

    public $checkifactive;
    public $user;

    public $page_id; //subcard id, used in query

    // public $referrals;

    public $query;
    public $referral_count_by_month;
    public $months;

    public $total_earning;
    public $period_earning;
    public $referral_total;

    public function getReferrals()
    {
        if ($this->affiliate_functionality == false) {
            abort(403, 'Access denied.');
        }

        $referQuery = Referral::from('referrals as refer')
            ->join('users', 'refer.referred_id', '=', 'users.id')
            ->select([
                'refer.referrer_id_main as referrer_id',
                'refer.referrer_id_sub as referrer_id_sub',
                'refer.referrer_type',
                'users.id as user_id',
                'users.username',
                'users.firstname as firstname',
                'users.lastname as lastname',
                'users.profile_pic',
                'users.job_title',
                'users.created_at as created_at',
            ]);

        // Apply correct referrer type
        if (isset($this->query['referrer_id_main'])) {
            $referQuery->where('refer.referrer_type', 'main')
                ->where('refer.referrer_id_main', $this->query['referrer_id_main']);
        } elseif (isset($this->query['referrer_id_sub'])) {
            $referQuery->where('refer.referrer_type', 'sub')
                ->where('refer.referrer_id_sub', $this->query['referrer_id_sub']);
        }

        $paginated = $referQuery->orderBy('created_at', 'desc')->paginate(10);

        $this->referral_total = $paginated->total();
        $this->user_type = $this->user->user_type;

        return $paginated;
    }

    public function showAnalytics()
    {
        $year = Carbon::now()->year;
        $referralCounts = [];

        for ($month = 1; $month <= 12; $month++) {
            $referralQuery = Referral::whereMonth('created_at', $month)
                ->whereYear('created_at', $year);

            if (isset($this->query['referrer_id_main'])) {
                $referralQuery->where('referrer_type', 'main')
                    ->where('referrer_id_main', $this->query['referrer_id_main']);
            } elseif (isset($this->query['referrer_id_sub'])) {
                $referralQuery->where('referrer_type', 'sub')
                    ->where('referrer_id_sub', $this->query['referrer_id_sub']);
            }

            $referralCounts[Carbon::create()->month($month)->format('M')] = $referralQuery->count();
        }

        $this->months = array_keys($referralCounts);
        $this->referral_count_by_month = $referralCounts;
    }

    public function updatingPage($value)
    {
        $this->dispatch('referralsUpdated');
    }

    public function mount()
    {
        $this->user = auth()->user();

        $this->checkifactive = User::where('id', $this->user->id)
            ->where('active_card', true)->count();

        if ($this->checkifactive > 0) {
            $this->query = ['referrer_id_main' => $this->user->id];
        } else {
            $this->page_id = SubCard::where('user_id', $this->user->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();
            $this->query = ['referrer_id_sub' => $this->page_id];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        $this->affiliate_functionality = $default->affiliate_functionality;

        $this->getReferrals();
        $this->showAnalytics();
    }

    public function render()
    {
        $referrals = $this->getReferrals();
        return view('livewire.referral-page', compact('referrals'))
            ->title('Referrals | ' . $this->site_name)->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
