<?php

namespace App\Livewire;

use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\Title;
use Illuminate\Support\Facades\Auth;
use App\Settings\CardDefaultsSetting;
use App\Helpers\SavingLinkHelpers;
use App\Models\SubCard;
use App\Models\UserSettings;
use Illuminate\Support\Facades\Http;

class ProfileEdit extends Component
{
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $user;
    public $user_type;
    public $user_type_get;
    public $username;
    public $firstname;
    public $lastname;
    public $job_title;
    public $company;
    public $country;
    public $state;
    public $company_website;
    public $bio;
    public $bio_long;
    public $phone;
    public $page_id;
    public $maincard_active; //used to determine which card is active

    //layout
    public $set_layout;
    public $check_cover;
    public $condition;
    public $availableStates = [];

    protected $rules = [
        'firstname'         => 'required|string|max:50',
        'lastname'          => 'required|string|max:50',
        'phone'             => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
        'job_title'         => 'required|string|max:200',
        'bio'               => 'nullable|string|max:200',
        'bio_long'          => 'nullable|string|max:10000',
        'company'           => 'nullable|string|max:255',
        'company_website'   => 'nullable|string|max:255',
        'country'           => 'required|string',
        'state'             => 'nullable|string',
    ];

    public function updatedCountry($value)
    {
        $this->state = null;
        $this->loadStates($value);
    }

    public function viewProfile()
    {
        // Check if main card is active
        $checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();
        // if active, get the datas and set to form
        if ($checkifactive > 0) {
            $this->maincard_active = true;
            $this->user = User::where('id', Auth::user()->id)
                ->select('id', 'username', 'firstname', 'lastname', 'phone', 'job_title', 'bio', 'bio_long', 'company', 'company_website', 'user_type', 'country', 'state')
                ->first();

            $this->user_type_get = $this->user->user_type;

            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
            $table = User::class;
            $id = auth()->user()->id;
        } else {
            // if main card is not active, get the infos from subcard
            $this->maincard_active = false;
            $this->user = SubCard::join('users', 'subcards.user_id', '=', 'users.id')
                ->where('subcards.user_id', auth()->user()->id)
                ->where('subcards.active_card', true)
                ->select(
                    'subcards.id',
                    'subcards.username',
                    'subcards.firstname',
                    'subcards.lastname',
                    'subcards.phone',
                    'subcards.job_title',
                    'subcards.bio',
                    'subcards.bio_long',
                    'subcards.company',
                    'subcards.company_website',
                    'users.user_type',
                )->first();

            $this->user_type_get = $this->user->user_type;
            $this->page_id = $this->user->id;

            $this->condition = ['page_id' => $this->page_id];
            $table = SubCard::class;
            $id = $this->page_id;
        }

        $this->set_layout = UserSettings::where('setting_key', 'profile_layout')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->check_cover = $table::where('id', $id)->pluck('cover_pic')->first();
    }

    public function updateUserProfile()
    {
        $subcard = auth()->user()->subcards()->where('active_card', true)->first();
        $subcard_username = $subcard ? $subcard->username : null;

        if ($this->maincard_active) {
            if ($this->username !== auth()->user()->username) {
                $this->rules['username'] = [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_.]+$/',
                    'unique:users,username',
                    'unique:subcards,username',
                ];
                $this->rules['country'] = [
                    'required',
                    'string',
                ];
            } else {
                $this->rules['username'] = [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_.]+$/',
                ];
                $this->rules['country'] = [
                    'required',
                    'string',
                ];
            }
        } else {
            if ($this->username !== $subcard_username) {
                $this->rules['username'] = [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_.]+$/',
                    'unique:users,username',
                    'unique:subcards,username',
                ];
            } else {
                $this->rules['username'] = [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_.]+$/',
                ];
            }
        }

        $this->validate(
            $this->rules,
            [
                'username.regex' => 'The username may only contain letters, numbers, and underscores.',
                'phone.regex' => 'Please enter a valid phone number.',
            ]
        );

        if (auth()->user()) {
            try {
                // if main card
                if ($this->maincard_active) {
                    //Update the user data
                    $updated = auth()->user()->update([
                        'username' => $this->username,
                        'firstname' => $this->firstname,
                        'lastname' => $this->lastname,
                        'phone' => $this->phone,
                        'job_title' => $this->job_title,
                        'bio' => $this->bio,
                        'bio_long' => $this->bio_long,
                        'company' => $this->company,
                        'company_website' => $this->company_website,
                        'country' => $this->country,
                        'state' => $this->state,
                    ]);
                } else {
                    //if subcard
                    $updated = SubCard::where('id', $this->page_id)
                        ->where('user_id', auth()->user()->id)
                        ->firstOrFail();

                    $updated->update([
                        'username' => $this->username,
                        'firstname' => $this->firstname,
                        'lastname' => $this->lastname,
                        'phone' => $this->phone,
                        'job_title' => $this->job_title,
                        'bio' => $this->bio,
                        'bio_long' => $this->bio_long,
                        'company' => $this->company,
                        'company_website' => $this->company_website,
                    ]);
                }

                if ($updated) {
                    session()->flash('success', 'Profile successfully updated.');
                    return redirect()->route('profile.edit');
                }
            } catch (\InvalidArgumentException $e) {
                session()->flash('company_website', $e->getMessage());
            }
        } else {
            session()->flash('error', 'Unauthorized action.');
            return redirect()->route('profile.edit');
        }
    }

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_url = $default->site_url;
        $this->site_name = $default->site_name;
        $this->app_name = $default->app_name;

        // this is used to set the old values of the fields
        $this->viewProfile();

        $this->username = $this->user->username;
        $this->firstname = $this->user->firstname;
        $this->lastname = $this->user->lastname;
        $this->phone = $this->user->phone;
        $this->job_title = $this->user->job_title;
        $this->bio = $this->user->bio;
        $this->bio_long = $this->user->bio_long;
        $this->user_type = $this->user_type_get;

        $this->company = $this->user->company;
        $this->country = $this->user->country;
        $this->state = $this->user->state;
        $this->company_website = $this->user->company_website;
        $this->thumbnail = $default->thumbnail;

        if ($this->country) {
            $this->loadStates($this->country);
        }
    }

    public function loadStates($country)
    {
        if (!$country) {
            $this->availableStates = [];
            return;
        }

        $response = Http::post('https://countriesnow.space/api/v0.1/countries/states', [
            'country' => $country,
        ]);

        $states = data_get($response->json(), 'data.states', []);

        $this->availableStates = collect($states)
            ->pluck('name', 'name')
            ->toArray();
    }

    public function render()
    {
        return view('pages.profile-edit')
            ->title('Edit Profile | ' . $this->site_name)->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
