<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductCollection;
use App\Models\ProductMedia;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class ProductEdit extends Component
{
    use AuthorizesRequests;

    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $product, $product_name, $stock, $shop_online_button, $shop_online_link, $description, $price;
    public $featured_image, $old_image, $new_product_image;
    public $productId;

    public $is_on_deal, $deal_starts_at, $deal_ends_at, $deal_price, $discount_percentage;
    public $deal_type = 'price';

    public $type, $sizesInput, $stylesInput, $colors = [];
    public $category_id = [];
    public $collection_id = [];

    public $product_videos;
    public $product_images;

    public $page_id, $condition;
    public $categories;
    public $collections;

    #[On('productImage')]
    public function setProductImage($productImageUrl)
    {
        $this->new_product_image = new TemporaryUploadedFile($productImageUrl, config('filesystems.default'));
    }

    public function addColor()
    {
        $this->colors[] = [
            'name' => '',
            'value' => '#000000',
        ];
    }

    public function removeColor($index)
    {
        unset($this->colors[$index]);
        // Re-index array to prevent gaps
        $this->colors = array_values($this->colors);
    }

    public function updateProduct()
    {
        try {
            $validated = $this->validate([
                'product_name'          => 'required|string',
                'price'                 => 'required|numeric',
                'is_on_deal'            => 'nullable|boolean',
                'deal_starts_at'        => 'nullable|date',
                'deal_ends_at'          => 'nullable|date',
                'stock'                 => 'required|numeric',
                'deal_price'            => 'nullable|numeric',
                'discount_percentage'   => 'nullable|numeric',
                'shop_online_button'    => 'nullable|string',
                'shop_online_link'      => 'nullable|url',
                'description'           => 'required|string|max:5000',
                'type'                  => 'required|string|in:physical,digital',
                // 'category_id'           => 'nullable|exists:product_categories,id',
                // 'collection_id'         => 'nullable|exists:products_collection,id',

                'category_id'   => 'nullable|array',
                'category_id.*' => 'exists:product_categories,id',

                'collection_id'   => 'nullable|array',
                'collection_id.*' => 'exists:products_collection,id',
            ]);

            // $validated['category_id'] = $this->category_id ?: null;
            // $validated['collection_id'] = $this->collection_id ?: null;

            $validated['category_id'] = empty($this->category_id)
                ? null
                : array_map('intval', $this->category_id);

            $validated['collection_id'] = empty($this->collection_id)
                ? null
                : array_map('intval', $this->collection_id);

            if ($this->new_product_image) {
                if ($this->old_image && file_exists(storage_path('app/public/' . $this->old_image))) {
                    unlink(storage_path('app/public/' . $this->old_image));
                }
                $validated['featured_image'] = $this->new_product_image->storePublicly('products', 'public');

                $tempFilePath = $this->new_product_image->getRealPath();
                if (file_exists($tempFilePath)) unlink($tempFilePath);
            }

            // Include type-specific data
            if ($this->type === 'physical') {

                // $sizes  = is_array($this->sizesInput) ? implode(',', $this->sizesInput) : $this->sizesInput;
                // $styles = is_array($this->stylesInput) ? implode(',', $this->stylesInput) : $this->stylesInput;

                // $validated['sizes']  = json_encode(array_map('trim', explode(',', $sizes)));
                // $validated['styles'] = json_encode(array_map('trim', explode(',', $styles)));
                // $validated['colors'] = json_encode($this->colors);

                // Sizes (ALWAYS clean empty values)
                $sizes = array_values(array_filter(
                    is_array($this->sizesInput)
                        ? $this->sizesInput
                        : array_map('trim', explode(',', (string) $this->sizesInput)),
                    fn($value) => trim($value) !== ''
                ));

                // Styles (ALWAYS clean empty values)
                $styles = array_values(array_filter(
                    is_array($this->stylesInput)
                        ? $this->stylesInput
                        : array_map('trim', explode(',', (string) $this->stylesInput)),
                    fn($value) => trim($value) !== ''
                ));

                $validated['sizes']  = json_encode($sizes);
                $validated['styles'] = json_encode($styles);
                $validated['colors'] = json_encode($this->colors);
            } else {
                $validated['sizes']  = json_encode([]);
                $validated['styles'] = json_encode([]);
                $validated['colors'] = null;
            }

            // dd($validated['sizes'], $validated['styles']);
            $update = Product::where('id', $this->productId)->update($validated);

            session()->flash($update ? 'success' : 'error', $update ? 'Product successfully updated!' : 'Unauthorized action.');

            return $this->redirect('/dashboard/product', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function viewProduct($id)
    {
        $this->product = Product::where($this->condition)
            ->where('id', $id)
            // ->select('id', 'product_name', 'description', 'stock', 'price', 'featured_image', 'shop_online_button', 'shop_online_link', 'type', 'sizes', 'styles', 'colors', 'category_id', 'collection_id')
            ->firstOrFail();

        // Populate edit fields
        $this->product_name         = $this->product->product_name;
        $this->description          = $this->product->description;
        $this->price                = $this->product->price;
        $this->is_on_deal           = $this->product->is_on_deal;
        $this->deal_starts_at       = $this->product->deal_starts_at
            ? \Carbon\Carbon::parse($this->product->deal_starts_at)->format('Y-m-d')
            : null;

        $this->deal_ends_at         = $this->product->deal_ends_at
            ? \Carbon\Carbon::parse($this->product->deal_ends_at)->format('Y-m-d')
            : null;

        $this->deal_price           = $this->product->deal_price;
        $this->discount_percentage  = $this->product->discount_percentage;
        $this->stock                = $this->product->stock;
        $this->featured_image       = $this->product->featured_image;
        $this->old_image            = $this->product->featured_image;
        $this->shop_online_button   = $this->product->shop_online_button ?? 'Shop Online';
        $this->shop_online_link     = $this->product->shop_online_link;
        $this->type                 = $this->product->type ?? 'physical';
        // dd($this->deal_starts_at);
        // $this->sizesInput   = $this->product->sizes;
        // $this->stylesInput  = $this->product->styles;

        // Sizes
        $this->sizesInput = is_string($this->product->sizes)
            ? implode(',', json_decode($this->product->sizes, true) ?? [])
            : $this->product->sizes;

        // Styles
        $this->stylesInput = is_string($this->product->styles)
            ? implode(',', json_decode($this->product->styles, true) ?? [])
            : $this->product->styles;

        // Fix colors issue
        if (is_array($this->product->colors)) {
            $this->colors = $this->product->colors;
        } elseif (is_string($this->product->colors)) {
            $this->colors = json_decode($this->product->colors, true) ?? [];
        } else {
            $this->colors = [];
        }

        // $this->category_id  = $this->product->category_id;
        // $this->collection_id  = $this->product->collection_id;

        $this->category_id = is_array($this->product->category_id)
            ? array_map('intval', $this->product->category_id)
            : (is_string($this->product->category_id)
                ? array_map('intval', json_decode($this->product->category_id, true) ?? [])
                : []);

        $this->collection_id = is_array($this->product->collection_id)
            ? array_map('intval', $this->product->collection_id)
            : (is_string($this->product->collection_id)
                ? array_map('intval', json_decode($this->product->collection_id, true) ?? [])
                : []);


        // Load categories for dropdown
        $this->categories = ProductCategory::where($this->condition)->get();
        $this->collections = ProductCollection::where($this->condition)->get();
    }


    public function mount($id)
    {
        $checkIfActive = User::where('id', auth()->id())->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())->where('active_card', true)->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->id(), 'page_id' => NULL];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->productId = $id;
        $this->viewProduct($this->productId);

        $this->product_videos = ProductMedia::where('product_id', $this->productId)->where('type', 'video')->count();
        $this->product_images = ProductMedia::where('product_id', $this->productId)->where('type', 'image')->count();

        $this->dispatch('setEditProductImage', $this->featured_image);
    }

    public function render()
    {
        $this->authorize('update', Product::class);

        return view('pages.product-edit')
            ->title('Edit Product | ' . $this->site_name)
            ->layoutData(['thumbnail' => $this->thumbnail, 'app_name' => $this->app_name,]);
    }
}
