<?php
// app/Livewire/NotificationsPage.php

namespace App\Livewire;

use App\Models\SubCard;
use Livewire\Component;
use Livewire\WithPagination;
use App\Settings\CardDefaultsSetting;
use Carbon\Carbon;

class NotificationsPage extends Component
{
    use WithPagination;

    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    protected $paginationTheme = 'tailwind';
    public $checkifactive;
    public $page_id;
    public $notification_count, $unread_notification_count;

    protected $listeners = [
        'notificationAdded' => '$refresh',
    ];

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
    }

    public function openNotification(string $id)
    {
        $activeSubCard = SubCard::where('user_id', auth()->id())
            ->where('active_card', true)
            ->first();

        if ($activeSubCard) {
            // Fetch notifications for SubCard
            $notification = $activeSubCard->notifications()
                ->where('id', $id)
                ->firstOrFail();
        } else {
            // Fetch notifications for main user
            $notification = auth()->user()
                ->notifications()
                ->where('id', $id)
                ->firstOrFail();
        }

        if (! $notification->read_at) {
            $notification->markAsRead();
        }

        return redirect()->to($notification->data['url'] ?? route('notifications.index'));
    }

    public function clearAllNotifications()
    {
        $activeSubCard = SubCard::where('user_id', auth()->id())
            ->where('active_card', true)
            ->first();

        if ($activeSubCard) {
            // Fetch notifications for SubCard
            $activeSubCard->notifications()
                ->delete();
        } else {
            // Fetch notifications for main user
            auth()->user()
                ->notifications()
                ->delete();
        }

        // // Delete all notifications for the user
        // auth()->user()->notifications()->delete();

        // clear read notifications and keep unread ones
        // auth()->user()->readNotifications()->delete();

        return $this->redirect(route('notifications.index'), navigate: true);
    }

    public function markAllAsRead()
    {
        $activeSubCard = SubCard::where('user_id', auth()->id())
            ->where('active_card', true)
            ->first();

        if ($activeSubCard) {
            // Mark all SubCard notifications as read
            $activeSubCard->unreadNotifications->markAsRead();
        } else {
            // Mark all main user notifications as read
            auth()->user()->unreadNotifications->markAsRead();
        }
    }

    public function render()
    {
        $activeSubCardId = SubCard::where('user_id', auth()->id())
            ->where('active_card', true)
            ->pluck('id')
            ->first();

        if ($activeSubCardId) {
            // Fetch notifications for SubCard
            $notificationsQuery = \App\Models\SubCard::find($activeSubCardId)
                ->notifications()
                ->latest();

            $this->unread_notification_count = \App\Models\SubCard::find($activeSubCardId)->unreadNotifications->count();
        } else {
            // Fetch notifications for main user
            $notificationsQuery = auth()->user()
                ->notifications()
                ->latest();

            $this->unread_notification_count =  auth()->user()->unreadNotifications->count();
        }

        $notifications = $notificationsQuery->paginate(10);
        $this->notification_count = $notificationsQuery->count();

        // dd($this->unread_notification_count);

        // Group notifications by date
        $groups = [
            'Today' => [],
            'Yesterday' => [],
            'Earlier' => [],
        ];

        foreach ($notifications as $notification) {
            $created = \Carbon\Carbon::parse($notification->created_at);

            if ($created->isToday()) {
                $groups['Today'][] = $notification;
            } elseif ($created->isYesterday()) {
                $groups['Yesterday'][] = $notification;
            } else {
                $groups['Earlier'][] = $notification;
            }
        }

        return view('livewire.notifications-page', [
            'notifications' => $notifications,
            'groups' => $groups,
        ])->title('Notifications | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
