<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Button;
use App\Models\Member;
use App\Models\LeadCaptureForm;
use App\Models\FormSubmission;
use App\Models\User;
use App\Models\SubCard;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;

class FormEditPage extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $page_id, $condition, $checkifactive;

    public
        $form,
        $title,
        $description,
        $active_image,
        $video,
        $active_video,
        $sender_email,
        $email_received,
        $link_attached,
        $is_active;
    public $form_id;
    public $form_submissions;
    public $new_image, $old_image;

    #[On('galleryImage')]
    public function setImage($newImageUrl)
    {
        $this->new_image = new TemporaryUploadedFile($newImageUrl, config('filesystems.default'));
    }

    public function updateForm()
    {
        try {
            $validated = $this->validate([
                'title'             => 'required|string',
                'description'       => 'nullable|string|max:500',
                'active_image'      => 'required|boolean',
                'video'             => 'nullable|string',
                'active_video'      => 'required|boolean',
                'sender_email'      => 'required|email:filter',
                'email_received'    => 'nullable|string|max:1000',
                'link_attached'     => 'required|string',
                'is_active'         => 'required|boolean',
            ]);

            if ($this->new_image) {
                // Delete the old thumbnail if it exists
                if ($this->old_image && Storage::disk('public')->exists($this->old_image)) {
                    Storage::disk('public')->delete($this->old_image);
                }

                // Store the new image
                $validated['image'] = $this->new_image->storePublicly('images', 'public');

                // Delete temp file if still there (optional, Laravel usually cleans this up)
                $tempFilePath = $this->new_image->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);
                }
            }

            if ($this->checkifactive < 1) {
                $validated['page_id'] = $this->page_id;
            } else {
                $validated['user_id'] = auth()->user()->id;
            }

            $update = LeadCaptureForm::findOrFail($this->form_id);

            $update->fill($validated);
            $update->save();

            session()->flash('success', 'Form successfully updated!');

            return $this->redirectRoute('form.edit', [
                'id' => $this->form_id,
            ], navigate: true);
        } catch (ValidationException $e) {
            dd($e);
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        } catch (\Throwable $e) {
            $this->dispatch('set-submitting', ['value' => false]);
            session()->flash('error', 'Something went wrong while updating the team.');
            return redirect()->back()->withInput();
        }
    }

    public function mount($id)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->form_id = $id;

        $this->checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        $this->form = LeadCaptureForm::where($this->condition)
            ->where('id', $this->form_id)
            ->first();

        $this->form_submissions = FormSubmission::where('form_id', $this->form_id)->get();

        if (!$this->form) {
            abort(403, 'Access denied');
        }

        $this->title = $this->form->title;
        $this->description = $this->form->description;
        $this->active_image = $this->form->active_image;
        $this->video = $this->form->video;
        $this->active_video = $this->form->active_video;
        $this->sender_email = $this->form->sender_email;
        $this->email_received = $this->form->email_received;
        $this->link_attached = $this->form->link_attached;
        $this->is_active = $this->form->is_active;

        $this->old_image = $this->form->image;

        $this->dispatch('setEditImage', $this->old_image);
    }

    public function render()
    {
        $this->authorize('update', LeadCaptureForm::class);
        return view('livewire.form-edit-page')->title('Update Form | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'form_submissions' => $this->form_submissions,
            'app_name' => $this->app_name,
        ]);
    }
}
