<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\LeadCaptureForm;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class FormCreatePage extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public
        $title,
        $description,
        $image,
        $active_image = true,
        $video,
        $active_video = true,
        $sender_email,
        $email_received = 'We received your submission. Please click the link below to join the zoom call.',
        $link_attached,
        $is_active = true;

    public $page_id; //subcard id, used in query

    // public function toggleActive()
    // {
    //     $this->is_active = !$this->is_active;
    // }

    #[On('galleryImage')]
    public function setImage($newImageUrl)
    {
        $this->image = new TemporaryUploadedFile($newImageUrl, config('filesystems.default'));
    }

    public function createForm()
    {
        try {
            $validated = $this->validate([
                'image'             => 'nullable|image|max:2048',
                'title'             => 'required|string',
                'description'       => 'nullable|string|max:500',
                'active_image'      => 'required|boolean',
                'video'             => 'nullable|string',
                'active_video'      => 'required|boolean',
                'sender_email'      => 'required|email:filter',
                'email_received'    => 'nullable|string|max:1000',
                'link_attached'     => 'required|string',
                'is_active'         => 'required|boolean',
            ]);

            $checkifactive = User::where('id', auth()->user()->id)
                ->where('active_card', true)->count();

            // $validated['type'] = 'team';
            // $validated['view'] = 'links';
            $validated['user_id'] = auth()->user()->id;

            //if maincard is not active
            if ($checkifactive < 1) {
                $this->page_id = SubCard::where('user_id', auth()->user()->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();

                $validated['page_id'] = $this->page_id;
            }

            if ($this->image) {
                // Store the new image
                $validated['image'] = $this->image->storePublicly('images', 'public');

                // Delete temp file if still there (optional, Laravel usually cleans this up)
                $tempFilePath = $this->image->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);
                }
            }

            $createNewForm = LeadCaptureForm::create($validated);

            if ($createNewForm) {
                session()->flash('success', 'Form successfully added!');
                return $this->redirectRoute('form.edit', [
                    'id' => $createNewForm->id,
                ], navigate: true);
            } else {
                session()->flash('error', 'Unauthorized action.');
            }
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function mount()
    {
        $this->sender_email = auth()->user()->email;
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->authorize('create', LeadCaptureForm::class);
        return view('livewire.form-create-page')->title('Create New Form | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
