<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Button;
use App\Models\Member;
use App\Models\User;
use App\Models\SubCard;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;

class EditTeamPage extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $page_id, $condition, $checkifactive;

    public $team, $title, $description, $is_active;
    public $team_id;
    public $members;
    public $new_thumbnail, $old_thumbnail;

    #[On('galleryImage')]
    public function setImage($newImageUrl)
    {
        $this->new_thumbnail = new TemporaryUploadedFile($newImageUrl, config('filesystems.default'));
    }

    public function updateTeam()
    {
        try {
            $validated = $this->validate([
                'title'         => 'required|string',
                // 'description'   => 'nullable|string',
                'description'   => 'nullable|string|max:1000',
                'is_active'     => 'required|boolean',
            ]);

            if ($this->new_thumbnail) {
                // Delete the old thumbnail if it exists
                if ($this->old_thumbnail && Storage::disk('public')->exists($this->old_thumbnail)) {
                    Storage::disk('public')->delete($this->old_thumbnail);
                }

                // Store the new image
                $validated['thumbnail'] = $this->new_thumbnail->storePublicly('thumbnails', 'public');

                // Delete temp file if still there (optional, Laravel usually cleans this up)
                $tempFilePath = $this->new_thumbnail->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);
                }
            }

            if ($this->checkifactive < 1) {
                $validated['page_id'] = $this->page_id;
            } else {
                $validated['user_id'] = auth()->user()->id;
            }

            // $newImage = SaleImage::findOrFail($this->image);
            $update = Button::findOrFail($this->team_id);

            $update->fill($validated);
            $update->save();

            session()->flash('success', 'Team successfully updated!');

            return $this->redirectRoute('team.edit', [
                'id' => $this->team_id,
            ], navigate: true);
        } catch (ValidationException $e) {
            // dd($e);
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        } catch (\Throwable $e) {
            $this->dispatch('set-submitting', ['value' => false]);
            session()->flash('error', 'Something went wrong while updating the team.');
            return redirect()->back()->withInput();
        }
    }

    public function mount($id)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;


        $this->team_id = $id;

        $this->checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        $this->team = Button::where($this->condition)
            ->where('id', $this->team_id)
            ->select('id', 'title', 'description', 'is_active', 'thumbnail')
            ->first();

        if (!$this->team) {
            abort(403, 'Access denied');
        }

        // dd($this->team);
        $networksMain = Member::join('users', 'members.member_id', '=', 'users.id')
            ->where('members.card_type', 'main')
            // ->where($networkCondition)
            ->where('members.team_id', $this->team_id)
            ->select([
                'members.id as member_id',
                'members.card_type',
                'members.member_id as entity_id',
                'members.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.profile_pic',
                'users.job_title',
                DB::raw('"main" as entity_type'),
            ]);

        $networksSub = Member::join('subcards', 'members.member_id', '=', 'subcards.id')
            ->where('members.card_type', 'sub')
            // ->where($networkCondition)
            ->where('members.team_id', $this->team_id)
            ->select([
                'members.id as member_id',
                'members.card_type',
                'members.member_id as entity_id',
                'members.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.profile_pic',
                'subcards.job_title',
                DB::raw('"sub" as entity_type'),
            ]);

        $this->members = $networksMain->union($networksSub)->orderBy('order')->get();

        $this->title = $this->team->title;
        $this->description = $this->team->description;
        $this->is_active = $this->team->is_active;
        $this->old_thumbnail = $this->team->thumbnail;

        $this->dispatch('setEditImage', $this->old_thumbnail);
    }

    public function render()
    {
        // $this->authorize('update', Button::class);
        return view('livewire.edit-team-page')->title('Edit Team Page | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'members' => $this->members,
            'app_name' => $this->app_name,
        ]);
    }
}
