<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductCollection;
use App\Models\ProductMedia;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class EditSubscriptionProduct extends Component
{
    use AuthorizesRequests;

    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $product_name,
        $price,
        $coin,
        $shop_online_link,
        $description,
        $has_packages,
        $has_coins;

    public array $subscription_options = [];
    public $featured_image, $old_image, $new_product_image;
    public $productId;

    public $type;
    public $product;
    public $product_videos;
    public $product_images;

    public $page_id, $condition;

    public function toggleHasPackages()
    {
        $this->has_packages = !$this->has_packages;
    }

    public function toggleHasCoins()
    {
        $this->has_coins = !$this->has_coins;
    }

    public function addProduct()
    {
        $this->subscription_options[] = [
            'title' => '',
            'price' => '',
            'coin' => '',
            'subscription_link' => '',
            'tier_description' => '',
        ];
    }

    public function removeProduct($index)
    {
        unset($this->subscription_options[$index]);
        $this->subscription_options = array_values($this->subscription_options); // reindex
    }

    #[On('productImage')]
    public function setProductImage($productImageUrl)
    {
        $this->new_product_image = new TemporaryUploadedFile($productImageUrl, config('filesystems.default'));
    }

    public function updateProduct()
    {
        try {
            $validated = $this->validate([
                'product_name'                              => 'required|string',
                'price'                                     => 'nullable|numeric',
                'coin'                                      => 'nullable|numeric',
                'shop_online_link'                          => 'nullable|url',
                'description'                               => 'required|string|max:5000',
                'subscription_options'                      => 'nullable|array',
                'subscription_options.*.title'              => 'nullable|string|max:255',
                'subscription_options.*.price'              => 'nullable|numeric',
                'subscription_options.*.coin'               => 'nullable|numeric',
                'subscription_options.*.subscription_link'  => 'nullable|url',
                'subscription_options.*.tier_description'   => 'nullable|string|max:5000',
                'has_packages'                              => 'nullable|boolean',
                'has_coins'                                 => 'nullable|boolean'
            ]);

            Log::info('Validated data', $validated);

            if ($this->new_product_image) {
                if ($this->old_image && file_exists(storage_path('app/public/' . $this->old_image))) {
                    unlink(storage_path('app/public/' . $this->old_image));
                }
                $validated['featured_image'] = $this->new_product_image->storePublicly('products', 'public');

                $tempFilePath = $this->new_product_image->getRealPath();
                if (file_exists($tempFilePath)) unlink($tempFilePath);
            }

            // Encode subscription_options as JSON if has_packages is true
            if ($this->has_packages && !empty($this->subscription_options)) {
                $validated['subscription_options'] = $this->subscription_options ?? [];
                $validated['price'] = null;
                $validated['coin'] = null;
                $validated['shop_online_link'] = null;
            } else {
                $validated['subscription_options'] = null;
            }

            $validated['has_packages'] = $this->has_packages;

            // Log::info('Final data to update', $validated);

            $update = auth()->user()->products()->where('id', $this->productId)->update($validated);

            session()->flash($update ? 'success' : 'error', $update ? 'Subscription product successfully updated!' : 'Unauthorized action.');

            return $this->redirect('/dashboard/subscriptions', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function mount($id)
    {
        $checkIfActive = User::where('id', auth()->id())->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())->where('active_card', true)->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->id(), 'page_id' => null];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->productId = $id;

        $product = Product::where($this->condition)
            ->where('id', $this->productId)
            // ->select('id', 'product_name', 'description', 'price', 'featured_image', 'shop_online_button', 'shop_online_link', 'type', 'has_packages', 'subscription_options')
            ->firstOrFail();

        // Populate basic fields
        $this->product_name         = $product->product_name;
        $this->description          = $product->description;
        $this->price                = $product->price;
        $this->coin                 = $product->coin;
        $this->shop_online_link     = $product->shop_online_link;
        $this->has_packages         = $product->has_packages ?? false;
        $this->has_coins            = $product->has_coins ?? false;
        $this->featured_image       = $product->featured_image;
        $this->old_image            = $product->featured_image;

        $this->subscription_options = $product->subscription_options ?? [];

        // Ensure at least one empty subscription option if has_packages is true but no options exist
        if ($this->has_packages && empty($this->subscription_options)) {
            $this->subscription_options[] = [
                'title'             => '',
                'price'             => '',
                'coin'              => '',
                'subscription_link' => '',
                'tier_description'  => '',
            ];
        }

        // Count media for galleries
        $this->product_videos = ProductMedia::where('product_id', $this->productId)->where('type', 'video')->count();
        $this->product_images = ProductMedia::where('product_id', $this->productId)->where('type', 'image')->count();

        // Send featured image to Livewire image picker
        $this->dispatch('setEditProductImage', $this->featured_image);
    }

    public function render()
    {
        $this->authorize('update', Product::class);

        return view('livewire.edit-subscription-product')
            ->title('Edit Subscription Product | ' . $this->site_name)
            ->layoutData(['thumbnail' => $this->thumbnail, 'app_name' => $this->app_name,]);
    }
}
