<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\SaleImage;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class EditSalesImage extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;

    public $old_image, $new_sales_image = null;

    public SaleImage $image;

    public $product_name, $description, $stock, $price, $image_path, $shop_online_link, $sales_image;
    public $is_active = true;
    public $shop_online_button = "Shop";
    public $sales_id;

    public $eventId;

    #[On('salesImage')]
    public function setSalesImage($salesImageUrl)
    {
        $this->new_sales_image = new TemporaryUploadedFile(
            $salesImageUrl,
            config('filesystems.default')
        );
    }

    public function updateImage()
    {
        try {
            $validated = $this->validate([
                // 'image'                 => 'required|image|max:2048',
                'product_name'          => 'required|string|max:100',
                'stock'                 => 'required|numeric',
                'price'                 => 'required|numeric',
                'description'           => 'nullable|string|max:300',
                'shop_online_button'    => 'nullable|string',
                'shop_online_link'      => 'required|url',
                'is_active'             => 'required|boolean',
            ]);

            // $newImage = SaleImage::findOrFail($this->image);
            $newImage = SaleImage::findOrFail($this->image->id);

            // Handle new image
            if ($this->new_sales_image) {
                if ($this->old_image && Storage::disk('public')->exists($this->old_image)) {
                    Storage::disk('public')->delete($this->old_image);
                }
                $validated['image_path'] = $this->new_sales_image->storePublicly('sales', 'public');
            }

            $newImage->fill($validated);
            $newImage->save();

            session()->flash('success', 'Image successfully updated!');

            return $this->redirectRoute('sales.edit', [
                'id' => $this->sales_id,
            ], navigate: true);
        } catch (ValidationException $e) {
            throw $e;
        } catch (\Throwable $e) {
            session()->flash('error', 'Something went wrong while updating the image.');
            return redirect()->back()->withInput();
        }
    }

    public function toggleActive()
    {
        $this->is_active = !$this->is_active;
    }

    public function mount(SaleImage $image)
    {
        $this->image = $image;
        $this->sales_id = $image->sales_id;

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;

        $this->image_path = $image->image_path;
        $this->product_name = $image->product_name;
        $this->description = $image->description;
        $this->stock = $image->stock;
        $this->price = $image->price;
        $this->shop_online_button = $image->shop_online_button;
        $this->shop_online_link = $image->shop_online_link;
        $this->is_active = $image->is_active;
        $this->sales_image = $image->image_path;
        $this->old_image =  $image->image_path;

        $this->dispatch('setEditSalesImage', $this->image_path);
    }

    public function render()
    {
        $this->authorize('update', SaleImage::class);
        return view('livewire.edit-sales-image')
            ->title('Edit Image | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
            ]);
    }
}
