<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\ProductCollection;
use App\Models\SubCard;
use App\Models\User;
use App\Settings\CardDefaultsSetting;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Livewire\Attributes\On;
use Illuminate\Validation\ValidationException;

class EditProductCollection extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $image, $title, $description;
    public $new_collection_image;

    public $page_id; //subcard id, used in query
    public $condition;
    public $collectionId;
    public $collection;

    #[On('bannerImage')]
    public function setBannerImage($adImageUrl)
    {
        $this->new_collection_image = new TemporaryUploadedFile($adImageUrl, config('filesystems.default'));
    }

    public function viewCollection($id)
    {
        $this->collection = ProductCollection::where($this->condition)
            ->where('id', $id)
            ->select('id', 'image', 'title', 'description')
            ->first();

        if (!$this->collection) {
            abort(403, 'Access denied');
        }
    }

    public function mount($id)
    {
        $checkIfActive = User::where('id', auth()->id())
            ->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())
                ->where('active_card', true)
                ->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->collectionId = $id;
        $this->viewCollection($this->collectionId);
        $this->image = $this->collection->image;
        $this->title = $this->collection->title;
        $this->description = $this->collection->description;
        $this->dispatch('setEditBannerImage', $this->image);
    }

    public function updateCollection()
    {
        try {
            $validated = $this->validate([
                'title'       => 'required|string',
                'description' => 'nullable|string|max:500'
            ]);

            // check if ad has image
            if ($this->new_collection_image) {
                // Check if the ad has already a picture and delete it
                if ($this->image) {
                    unlink(storage_path('app/public/' . $this->image));
                }

                $validated['image'] = $this->new_collection_image->storePublicly('product_collection', 'public');

                $tempFilePath = $this->new_collection_image->getRealPath();

                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);  // Delete the temporary file explicitly
                }
            }

            $user = auth()->user();
            $checkMainActive = User::where('id', $user->id)
                ->where('active_card', true)
                ->exists();

            if ($checkMainActive) {
                $validated['user_id'] = $user->id;
            } else {
                $subCardId = SubCard::where('user_id', $user->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();

                if (!$subCardId) {
                    session()->flash('error', 'No active subcard found.');
                    return;
                }

                $validated['page_id'] = $subCardId;
            }

            $update = ProductCollection::where('id', $this->collectionId)->update($validated);

            if ($update) {
                session()->flash('success', 'Collection successfully updated!');
            } else {
                session()->flash('error', 'Failed to add collection.');
            }
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }

        return $this->redirect('/dashboard/product-collections', navigate: true);
    }

    public function render()
    {
        // $this->authorize('update', ProductCollection::class);
        return view('livewire.edit-product-collection')->title('Edit Collection | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
