<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\PresentationPage;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class EditPresentationPage extends Component
{
    use AuthorizesRequests, WithFileUploads;

    public $page_id;
    public $title;
    public $description;
    public $is_active = true;
    public $contents = [];
    public $blockOpen = []; // collapse states

    public $site_name;
    public $thumbnail;
    public $app_name;

    public function mount($id)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->page_id = $id;
        $page = PresentationPage::findOrFail($id);

        $this->title = $page->title;
        $this->description = $page->description;
        $this->is_active = $page->is_active;

        // Load blocks
        $this->contents = is_array($page->contents)
            ? $page->contents
            : json_decode($page->contents, true) ?? [];

        // Every block must have a file key
        foreach ($this->contents as $i => $block) {
            if (!isset($this->contents[$i]['file'])) {
                $this->contents[$i]['file'] = null;
            }
        }

        // All blocks open by default
        $this->blockOpen = array_fill(0, count($this->contents), true);
    }

    public function addBlock(string $type)
    {
        $this->contents[] = [
            'type' => $type,
            'title' => null,
            'value' => null,
            'file' => null,
        ];

        $this->blockOpen[] = true;
    }

    public function updateOrder(array $order)
    {
        $newContents = [];
        $newOpen = [];

        foreach ($order as $i) {
            $newContents[] = $this->contents[$i];
            $newOpen[] = $this->blockOpen[$i];
        }

        $this->contents = $newContents;
        $this->blockOpen = $newOpen;
    }

    public function removeBlock(int $index)
    {
        if (!isset($this->contents[$index])) return;

        unset($this->contents[$index]);
        unset($this->blockOpen[$index]);

        $this->contents = array_values($this->contents);
        $this->blockOpen = array_values($this->blockOpen);
    }

    public function moveUp(int $index)
    {
        if ($index <= 0) return;

        [$this->contents[$index - 1], $this->contents[$index]] =
            [$this->contents[$index], $this->contents[$index - 1]];

        [$this->blockOpen[$index - 1], $this->blockOpen[$index]] =
            [$this->blockOpen[$index], $this->blockOpen[$index - 1]];
    }

    public function moveDown(int $index)
    {
        if ($index >= count($this->contents) - 1) return;

        [$this->contents[$index + 1], $this->contents[$index]] =
            [$this->contents[$index], $this->contents[$index + 1]];

        [$this->blockOpen[$index + 1], $this->blockOpen[$index]] =
            [$this->blockOpen[$index], $this->blockOpen[$index + 1]];
    }

    protected function rules(): array
    {
        return [
            'title'                         => 'required|string|max:255',
            'description'                   => 'nullable|string|max:200',
            'is_active'                     => 'required|boolean',
            'contents'                      => 'nullable|array',
            'contents.*.type'               => 'required|string',
            'contents.*.title'              => 'nullable|string',
            'contents.*.value'              => 'nullable|string',
            'contents.*.file'               => 'nullable|image|max:2048',
            'contents.*.call_to_action'     => 'nullable|url',
        ];
    }

    public function savePage()
    {
        try {

            $this->validate();

            $processed = [];

            foreach ($this->contents as $i => $block) {
                $type = $block['type'] ?? null;

                if ($type === null) continue;

                switch ($type) {
                    case 'heading':
                    case 'html':
                    case 'video':
                        $value = trim($block['value'] ?? '');

                        if ($value === '') {
                            $this->addError("contents.$i.value", ucfirst($type) . " block cannot be empty.");
                            return;
                        }
                        if ($type === 'video') {
                            if (!filter_var($block['value'], FILTER_VALIDATE_URL)) {
                                $this->addError("contents.$i.value", "Please enter a valid URL.");
                                return;
                            }
                        }

                        $processed[] = [
                            'type' => $type,
                            'title' => $block['title'] ?? null,
                            'value' => $value,
                        ];
                        break;

                    case 'button':
                        $label = trim($block['title'] ?? '');
                        $url = trim($block['value'] ?? '');
                        // Initialize error flag
                        $hasError = false;

                        // 1. Required fields
                        if ($label === '') {
                            // session()->flash('error', "Button block (#" . ($i + 1) . ") requires a label and URL.");
                            $this->addError("contents.$i.title", "Button block requires a label.");
                            $hasError = true;
                        }

                        if ($url === '') {
                            // session()->flash('error', "Button block (#" . ($i + 1) . ") requires a label and URL.");
                            $this->addError("contents.$i.value", "Button block requires a URL.");
                            $hasError = true;
                        }

                        // 2. Check if valid URL
                        if (!filter_var($url, FILTER_VALIDATE_URL)) {
                            // session()->flash('error', "Button block (#" . ($i + 1) . ") must contain a valid URL.");
                            $this->addError("contents.$i.value", "Please enter a valid URL.");
                            $hasError = true;
                        }

                        // If any errors exist, stop processing
                        if ($hasError) {
                            return;
                        }

                        $processed[] = [
                            'type' => 'button',
                            'title' => $label,
                            'value' => $url,
                        ];
                        break;

                    case 'image':
                        $uploaded = $block['file'] ?? null;
                        $existing = $block['value'] ?? null;
                        $call_to_action = $block['call_to_action'] ?? null;

                        if ($uploaded instanceof TemporaryUploadedFile) {
                            $path = $uploaded->storePublicly('pages', 'public');
                            $processed[] = [
                                'type' => 'image',
                                'value' => $path,
                                'call_to_action' => $call_to_action,
                            ];
                        } elseif ($existing) {
                            $processed[] = [
                                'type' => 'image',
                                'value' => $existing,
                                'call_to_action' => $call_to_action,
                            ];
                        } else {
                            // session()->flash('error', "Image block (#" . ($i + 1) . ") requires an image.");
                            $this->addError("contents.$i.file", "Image block requires an uploaded file.");
                            return;
                        }
                        break;
                }
            }

            $page = PresentationPage::findOrFail($this->page_id);

            $page->update([
                'title' => $this->title,
                'description' => $this->description,
                'is_active' => $this->is_active,
                'contents' => $processed,
            ]);

            session()->flash('success', 'Page successfully updated!');

            return $this->redirect('/dashboard/pages', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        return view('livewire.edit-presentation-page')
            ->title('Edit Page | ' . $this->site_name)
            ->layoutData(['thumbnail' => $this->thumbnail, 'app_name' => $this->app_name,]);
    }
}
