<?php

namespace App\Livewire;

use App\Models\Button;
use Livewire\Component;
use App\Models\User;
use App\Models\SubCard;
use App\Models\UserSettings;
use App\Settings\CardDefaultsSetting;
use App\Models\TrustedNetwork;
use Illuminate\Support\Facades\DB;
use Livewire\WithPagination;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class DisplayTrustedNetwork extends Component
{
    use AuthorizesRequests;
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $card_title;
    public $page_title;
    public $app_name;
    public $username;
    public $user;
    public $fullname;
    public $favicon;

    public $trusted_network_title;

    public $checkMain;
    public $checkSub;
    public bool $mainCard = false;
    public $page_id;

    public $user_id;
    public $condition;

    public $color_settings_link;
    public $color_settings_custom_button;
    public $color_settings_custom_button_text;

    public $color_settings_profile_pic_border;
    public $color_settings_card_light;
    public $color_settings_card_dark;
    public $color_settings_text_light;
    public $color_settings_text_dark;

    //profile layout
    public $profile_layout;

    //style
    public $button_style;

    // Pagination & Search
    public $perPage = 15;
    public $keyword = '';

    public $owner_id;

    public function updatingKeyword()
    {
        $this->resetPage();
    }

    public function onKeywordUpdated()
    {
        // Always reset pagination when the search term changes
        $this->resetPage();

        // If the search bar is empty, force a full reset of members
        if (trim($this->keyword) === '') {
            $this->keyword = '';
            $this->resetPage();
        }
    }

    public function loadMore()
    {
        $this->perPage += 10;
    }

    public function getMembersProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // --- MAIN ---
        $networksMain = TrustedNetwork::join('users', 'trusted_network.member_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'main')
            ->where('trusted_network.is_accepted', true)
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"main" as entity_type'),
            ]);

        // --- SUB ---
        $networksSub = TrustedNetwork::join('subcards', 'trusted_network.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'sub')
            ->where('trusted_network.is_accepted', true)
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"sub" as entity_type'),
            ]);

        $unionQuery = $networksMain->unionAll($networksSub);

        return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
            ->mergeBindings($unionQuery->getQuery())
            ->orderBy('order')
            ->paginate($this->perPage);
    }

    public function getStateProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // determine current card owner's country (fallback to empty string)
        $myCountry = $this->user->country ?? '';

        // --- MAIN ---
        $networksMain = TrustedNetwork::join('users', 'trusted_network.member_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'main')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->where('users.country', $myCountry)
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.country',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('"main" as entity_type'),
            ]);

        // --- SUB ---
        $networksSub = TrustedNetwork::join('subcards', 'trusted_network.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'sub')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->where('users.country', $myCountry)
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'users.country',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('"sub" as entity_type'),
            ]);

        // union and get results
        $results = $networksMain->unionAll($networksSub)->get();

        // normalize and group by state (use fallback label for empty states)
        $grouped = $results->map(function ($r) {
            // ensure firstname/lastname exist on both main/sub selects (some rows may have username only)
            $r->firstname = $r->firstname ?? null;
            $r->lastname  = $r->lastname ?? null;
            $r->state     = $r->state ?? '';
            $r->country   = $r->country ?? '';
            $r->profile_pic = $r->profile_pic ?? '/img/profile-placeholder.jpg';
            return $r;
        })->groupBy(function ($item) {
            return trim($item->state) !== '' ? $item->state : ''; // use '—' for unknown/empty states
        });

        return $grouped;
    }

    public function getGlobalProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // Determine current card owner's country
        $myCountry = $this->user->country ?? '';

        // --- MAIN ---
        $networksMain = TrustedNetwork::join('users', 'trusted_network.member_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'main')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->where('users.country', '!=', $myCountry) // ← GLOBAL: exclude my country
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.country',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('"main" as entity_type'),
            ]);

        // --- SUB ---
        $networksSub = TrustedNetwork::join('subcards', 'trusted_network.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'sub')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->where('users.country', '!=', $myCountry) // ← GLOBAL: exclude my country
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'users.country',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('"sub" as entity_type'),
            ]);

        // union results
        $results = $networksMain->unionAll($networksSub)->get();

        // group by COUNTRY instead of state
        $grouped = $results->map(function ($r) {
            $r->firstname = $r->firstname ?? null;
            $r->lastname  = $r->lastname ?? null;
            $r->state     = $r->state ?? '';
            $r->country   = $r->country ?? '';
            $r->profile_pic = $r->profile_pic ?? '/img/profile-placeholder.jpg';
            return $r;
        })->groupBy(function ($item) {
            return trim($item->country) !== '' ? $item->country : 'Unknown';
        });

        return $grouped;
    }

    public function getEatProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // --- MAIN ---
        $networksMain = TrustedNetwork::join('users', 'trusted_network.member_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'main')
            ->where('trusted_network.type', 'eat')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"main" as entity_type'),
            ]);

        // --- SUB ---
        $networksSub = TrustedNetwork::join('subcards', 'trusted_network.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'sub')
            ->where('trusted_network.type', 'eat')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"sub" as entity_type'),
            ]);

        $unionQuery = $networksMain->unionAll($networksSub);

        return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
            ->mergeBindings($unionQuery->getQuery())
            ->orderBy('order')
            ->paginate($this->perPage);
    }

    public function getExpertProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // --- MAIN ---
        $networksMain = TrustedNetwork::join('users', 'trusted_network.member_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'main')
            ->where('trusted_network.type', 'expert')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"main" as entity_type'),
            ]);

        // --- SUB ---
        $networksSub = TrustedNetwork::join('subcards', 'trusted_network.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'sub')
            ->where('trusted_network.type', 'expert')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"sub" as entity_type'),
            ]);

        $unionQuery = $networksMain->unionAll($networksSub);

        return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
            ->mergeBindings($unionQuery->getQuery())
            ->orderBy('order')
            ->paginate($this->perPage);
    }

    public function getEventProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // --- MAIN ---
        $networksMain = TrustedNetwork::join('users', 'trusted_network.member_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'main')
            ->where('trusted_network.type', 'event')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"main" as entity_type'),
            ]);

        // --- SUB ---
        $networksSub = TrustedNetwork::join('subcards', 'trusted_network.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('trusted_network.card_type', 'sub')
            ->where('trusted_network.type', 'event')
            ->when(isset($this->condition['user_id']), fn($q) => $q->where('trusted_network.user_id', $this->condition['user_id']))
            ->when(isset($this->condition['page_id']), fn($q) => $q->where('trusted_network.page_id', $this->condition['page_id']))
            ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
                $query->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            }))
            ->select([
                'trusted_network.id as member_id',
                'trusted_network.member_id as entity_id',
                'trusted_network.card_type',
                'trusted_network.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"sub" as entity_type'),
            ]);

        $unionQuery = $networksMain->unionAll($networksSub);

        return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
            ->mergeBindings($unionQuery->getQuery())
            ->orderBy('order')
            ->paginate($this->perPage);
    }

    public function displayCard($username)
    {
        // get user creds
        $this->username = $username;

        if ($this->mainCard == true) {
            $this->user = User::where('username', $this->username)->where('is_active', true)->first(['email', 'phone', 'profile_pic', 'cover_pic', 'thumbnail', 'username', 'firstname', 'lastname', 'job_title', 'company', 'company_website', 'bio', 'language', 'address', 'country', 'id', 'user_type']);
        } else {
            $this->user = SubCard::join('users', 'subcards.user_id', '=', 'users.id')
                ->where('subcards.username', $this->username)
                ->first([
                    'subcards.id',
                    'subcards.user_id',
                    'subcards.username',
                    'subcards.job_title',
                    'subcards.company',
                    'subcards.company_website',
                    'subcards.bio',
                    'subcards.profile_pic',
                    'subcards.cover_pic',
                    'subcards.thumbnail as thumbnail',
                    'users.firstname',
                    'users.lastname',
                    'users.address',
                    'users.country',
                    'users.user_type',
                    'subcards.email',
                    'subcards.phone',
                    'users.id',
                ]);
        }

        if (!isset($this->username)) {
            $this->user = 0;
        }

        // If user exist
        if ($this->user) {
            // Set the fullanme
            $this->fullname = $this->user->firstname . ' ' . $this->user->lastname;

            // Page title
            $this->page_title = UserSettings::where('setting_key', 'custom_page_title')
                ->where($this->condition)
                ->pluck('setting_value')
                ->first();

            // Card title
            $this->card_title = UserSettings::where($this->condition)
                ->where('setting_key', 'card_custom_title')
                ->pluck('setting_value')
                ->first();

            $this->trusted_network_title = Button::where('type', 'trusted_network')
                ->where($this->condition)
                ->pluck('title')
                ->first();

            if (!$this->page_title) {
                $this->page_title = $this->card_title;
            }

            if (!$this->card_title) {
                $this->card_title = $this->fullname;
            }
        } else {
            $this->user = 0;
            $this->card_title = "Not found";
            $this->fullname = "Not found";
        }
    }

    public function mount($username)
    {
        $default = app(CardDefaultsSetting::class);

        // Default site values
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->favicon = $default->thumbnail;
        $this->app_name = $default->app_name;

        //used to determine where to check if the username exist
        $this->checkMain = User::where('username', $this->username)->count();
        $this->checkSub = SubCard::where('username', $this->username)->count();

        if ($this->checkMain > 0) {
            $this->mainCard = true;
            $this->user_id = User::where('username', $username)
                ->where('is_active', true)
                ->pluck('id')
                ->first();
            $this->condition = ['user_id' => $this->user_id, 'page_id' => NULL];
            $this->owner_id = $this->user_id;
        }

        if ($this->checkSub > 0) {
            $this->page_id = SubCard::where('username', $this->username)
                ->pluck('id')
                ->first();
            $this->mainCard = false;
            $this->condition = ['page_id' => $this->page_id];
            $this->owner_id = SubCard::where('username', $this->username)
                ->pluck('user_id')
                ->first();
        }

        // Get color and layout settings
        $this->color_settings_link = UserSettings::where('setting_key', 'color_settings_link')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_profile_pic_border = UserSettings::where('setting_key', 'color_settings_profile_pic_border')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_card_light = UserSettings::where('setting_key', 'color_settings_card_light')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_card_dark = UserSettings::where('setting_key', 'color_settings_card_dark')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->profile_layout = UserSettings::where('setting_key', 'profile_layout')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_text_light = UserSettings::where('setting_key', 'color_settings_text_light')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_text_dark = UserSettings::where('setting_key', 'color_settings_text_dark')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_custom_button = UserSettings::where('setting_key', 'color_settings_custom_button')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_custom_button_text = UserSettings::where('setting_key', 'color_settings_custom_button_text')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->button_style = UserSettings::where('setting_key', 'button_style')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        // Default values / fallbacks
        $this->profile_layout = $this->profile_layout ?? 'default';
        $this->color_settings_link = $this->color_settings_link ?: '#ca8a04';
        $this->color_settings_profile_pic_border = $this->color_settings_profile_pic_border ?: '#ca8a04';
        $this->color_settings_card_light = $this->color_settings_card_light ?: '#fff';
        $this->color_settings_card_dark = $this->color_settings_card_dark ?: '#18181b';
        $this->color_settings_text_light = $this->color_settings_text_light ?: '#1e293b';
        $this->color_settings_text_dark = $this->color_settings_text_dark ?: '#cbd5e1';
        $this->color_settings_custom_button_text = $this->color_settings_custom_button_text ?: '#000';
        $this->color_settings_custom_button = $this->color_settings_custom_button ?: '#facc15';

        //style
        $this->button_style = $this->button_style ?: 'rounded';

        $this->displayCard($username);
        // dd($this->user);
        if (!$this->user) {
            abort(404, 'Page not found.');
        }

        if ($this->user->thumbnail) {
            // User has a custom thumbnail
            $this->thumbnail = $this->user->thumbnail;
            $this->favicon = $this->user->thumbnail;
        } else if ($this->user->profile_pic) {
            // Use user's profile picture
            $this->thumbnail = $this->user->profile_pic;
            $this->favicon = $this->user->profile_pic;
        } else {
            // Default site thumbnail fallback
            $this->thumbnail = $this->thumbnail ?? null;
            $this->favicon = $this->thumbnail;
        }

        // dd($this->user, $this->user->thumbnail, $this->user->profile_pic, $this->thumbnail);
    }


    public function render()
    {
        if ($this->user->user_type != 'Platinum') {
            abort(404, 'Page not found.');
        }

        // $this->authorize('view', TrustedNetwork::class);
        return view('livewire.display-trusted-network', [
            'members' => $this->members,
            'state' => $this->state,
            'global' => $this->global,
            'eat' => $this->eat,
            'expert' => $this->expert,
            'event' => $this->event,
        ])->title($this->trusted_network_title . ' | ' . ($this->page_title ?: $this->card_title))->layoutData([
            'favicon' => $this->favicon,
            'thumbnail' => $this->thumbnail,
            'username' => $this->username,
            'site_url' => $this->site_url,
            'title' => $this->page_title ?: $this->card_title,
            'color_settings_card_light' => $this->color_settings_card_light,
            'color_settings_card_dark' => $this->color_settings_card_dark,
            'color_settings_text_light' => $this->color_settings_text_light,
            'color_settings_text_dark' => $this->color_settings_text_dark,
            'app_name' => $this->app_name,
        ]);
    }
}
