<?php

namespace App\Livewire;

use App\Models\Button;
use App\Models\Member;
use App\Models\User;
use App\Models\SubCard;
use App\Models\UserSettings;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithPagination;

class DisplayTeamPage extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    // Site & Page Settings
    public $site_url, $site_name, $thumbnail, $favicon, $app_name;
    public $username, $team_id, $team, $user, $fullname, $page_title, $card_title;
    public $color_settings_link, $color_settings_profile_pic_border;
    public $color_settings_card_light, $color_settings_card_dark;
    public $color_settings_text_light, $color_settings_text_dark;
    public $profile_layout;

    // Footer
    public ?bool $footer_enabled;
    public ?string $footer_text, $footer_link;
    public bool $customFooter = false;

    // Pagination & Search
    public $perPage = 15;
    public $keyword = '';

    // Card type
    public bool $mainCard = false;
    public $page_id;
    public $user_id;
    public $condition;

    public function mount($username, $team_id)
    {
        $default = app(CardDefaultsSetting::class);

        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        $this->username = $username;
        $this->team_id = $team_id;

        // Determine if main or sub card
        $checkMain = User::where('username', $username)->where('is_active', true)->exists();
        $checkSub = SubCard::where('username', $username)->exists();

        if ($checkMain) {
            $this->mainCard = true;
            $this->user_id = User::where('username', $username)->value('id');
            $this->condition = ['user_id' => $this->user_id, 'page_id' => null];
        } elseif ($checkSub) {
            $this->mainCard = false;
            $this->page_id = SubCard::where('username', $username)->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            abort(404, 'User not found');
        }

        $this->initializeSettings();
        $this->displayCard();
    }

    protected function initializeSettings()
    {
        $this->color_settings_link = UserSettings::where('setting_key', 'color_settings_link')
            ->where($this->condition)->value('setting_value') ?: '#ca8a04';
        $this->color_settings_profile_pic_border = UserSettings::where('setting_key', 'color_settings_profile_pic_border')
            ->where($this->condition)->value('setting_value') ?: '#ca8a04';
        $this->color_settings_card_light = UserSettings::where('setting_key', 'color_settings_card_light')
            ->where($this->condition)->value('setting_value') ?: '#fff';
        $this->color_settings_card_dark = UserSettings::where('setting_key', 'color_settings_card_dark')
            ->where($this->condition)->value('setting_value') ?: '#18181b';
        $this->color_settings_text_light = UserSettings::where('setting_key', 'color_settings_text_light')
            ->where($this->condition)->value('setting_value') ?: '#1e293b';
        $this->color_settings_text_dark = UserSettings::where('setting_key', 'color_settings_text_dark')
            ->where($this->condition)->value('setting_value') ?: '#cbd5e1';
        $this->profile_layout = UserSettings::where('setting_key', 'profile_layout')
            ->where($this->condition)->value('setting_value') ?: 'default';
    }

    protected function displayCard()
    {
        $this->user = $this->mainCard
            ? User::where('id', $this->user_id)->first()
            : SubCard::where('id', $this->page_id)->first();

        if (!$this->user) abort(404, 'User not found');

        $this->fullname = $this->user->firstname . ' ' . $this->user->lastname;
        $this->card_title = UserSettings::where('setting_key', 'card_custom_title')
            ->where($this->condition)->value('setting_value') ?: $this->fullname;
        $this->page_title = UserSettings::where('setting_key', 'custom_page_title')
            ->where($this->condition)->value('setting_value');
        $this->footer_enabled = UserSettings::where('setting_key', 'footer_enabled')
            ->where($this->condition)->value('setting_value') ?? true;
        $this->footer_text = UserSettings::where('setting_key', 'footer_text')
            ->where($this->condition)->value('setting_value');
        $this->footer_link = UserSettings::where('setting_key', 'footer_link')
            ->where($this->condition)->value('setting_value');
        $this->customFooter = $this->footer_text && $this->footer_link;

        $this->team = Button::where('id', $this->team_id)->where($this->condition)->first();
    }

    public function updatingKeyword()
    {
        $this->resetPage();
    }

    public function onKeywordUpdated()
    {
        // Always reset pagination when the search term changes
        $this->resetPage();

        // If the search bar is empty, force a full reset of members
        if (trim($this->keyword) === '') {
            $this->keyword = '';
            $this->resetPage();
        }
    }

    public function loadMore()
    {
        $this->perPage += 10;
    }

    // public function getMembersProperty()
    // {
    //     $keyword = '%' . $this->keyword . '%';

    //     $networksMain = Member::join('users', 'members.member_id', '=', 'users.id')
    //         ->where('members.card_type', 'main')
    //         ->where('members.team_id', $this->team_id)
    //         ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
    //             $query->where('users.username', 'like', $keyword)
    //                 ->orWhere('users.firstname', 'like', $keyword)
    //                 ->orWhere('users.lastname', 'like', $keyword)
    //                 ->orWhere('users.bio', 'like', $keyword)
    //                 ->orWhere('users.company', 'like', $keyword)
    //                 ->orWhere('users.country', 'like', $keyword)
    //                 ->orWhere('users.state', 'like', $keyword)
    //                 ->orWhere('users.job_title', 'like', $keyword);
    //         }))
    //         ->select([
    //             'members.id as member_id',
    //             'members.member_id as entity_id',
    //             'members.card_type',
    //             'members.custom_card_link',
    //             'members.order',
    //             'users.username',
    //             'users.firstname',
    //             'users.lastname',
    //             'users.job_title',
    //             'users.company',
    //             'users.state',
    //             'users.bio',
    //             DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
    //             DB::raw('COALESCE(users.country, "") as country'),
    //             DB::raw('"main" as entity_type'),
    //         ]);

    //     $networksSub = Member::join('subcards', 'members.member_id', '=', 'subcards.id')
    //         ->join('users', 'subcards.user_id', '=', 'users.id')
    //         ->where('members.card_type', 'sub')
    //         ->where('members.team_id', $this->team_id)
    //         ->when($this->keyword, fn($q) => $q->where(function ($query) use ($keyword) {
    //             $query->where('subcards.username', 'like', $keyword)
    //                 ->orWhere('subcards.firstname', 'like', $keyword)
    //                 ->orWhere('subcards.lastname', 'like', $keyword)
    //                 ->orWhere('subcards.company', 'like', $keyword)
    //                 ->orWhere('subcards.bio', 'like', $keyword)
    //                 ->orWhere('users.country', 'like', $keyword)
    //                 ->orWhere('users.state', 'like', $keyword)
    //                 ->orWhere('subcards.job_title', 'like', $keyword);
    //         }))
    //         ->select([
    //             'members.id as member_id',
    //             'members.member_id as entity_id',
    //             'members.card_type',
    //             'members.custom_card_link',
    //             'members.order',
    //             'subcards.username',
    //             'subcards.firstname',
    //             'subcards.lastname',
    //             'subcards.job_title',
    //             'subcards.company',
    //             'users.state',
    //             'subcards.bio',
    //             DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
    //             DB::raw('COALESCE(users.country, "") as country'),
    //             DB::raw('"sub" as entity_type'),
    //         ]);

    //     $unionQuery = $networksMain->unionAll($networksSub);

    //     return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
    //         ->mergeBindings($unionQuery->getQuery())
    //         ->orderBy('order')
    //         ->paginate($this->perPage);
    // }

    public function getMembersProperty()
    {
        $keyword = '%' . $this->keyword . '%';

        // Main card members
        $queryMain = Member::join('users', 'members.member_id', '=', 'users.id')
            ->where('members.card_type', 'main')
            ->where('members.team_id', $this->team_id)
            ->select([
                'members.id as member_id',
                'members.member_id as entity_id',
                'members.card_type',
                'members.custom_card_link',
                'members.order',
                'users.username',
                'users.firstname',
                'users.lastname',
                'users.job_title',
                'users.company',
                'users.state',
                'users.bio',
                DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"main" as entity_type'),
                DB::raw('(SELECT setting_value
                      FROM user_settings
                      WHERE user_id = users.id AND setting_key = "card_custom_title" AND page_id IS NULL
                      LIMIT 1) as card_custom_title'),
            ]);

        // Sub card members
        $querySub = Member::join('subcards', 'members.member_id', '=', 'subcards.id')
            ->join('users', 'subcards.user_id', '=', 'users.id')
            ->where('members.card_type', 'sub')
            ->where('members.team_id', $this->team_id)
            ->select([
                'members.id as member_id',
                'members.member_id as entity_id',
                'members.card_type',
                'members.custom_card_link',
                'members.order',
                'subcards.username',
                'subcards.firstname',
                'subcards.lastname',
                'subcards.job_title',
                'subcards.company',
                'users.state',
                'subcards.bio',
                DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
                DB::raw('COALESCE(users.country, "") as country'),
                DB::raw('"sub" as entity_type'),
                DB::raw('(SELECT setting_value
                      FROM user_settings
                      WHERE page_id = subcards.id AND setting_key = "card_custom_title"
                      LIMIT 1) as card_custom_title'),
            ]);

        // Apply keyword search if needed
        if ($this->keyword) {
            $queryMain->where(function ($q) use ($keyword) {
                $q->where('users.username', 'like', $keyword)
                    ->orWhere('users.firstname', 'like', $keyword)
                    ->orWhere('users.lastname', 'like', $keyword)
                    ->orWhere('users.bio', 'like', $keyword)
                    ->orWhere('users.company', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('users.job_title', 'like', $keyword);
            });

            $querySub->where(function ($q) use ($keyword) {
                $q->where('subcards.username', 'like', $keyword)
                    ->orWhere('subcards.firstname', 'like', $keyword)
                    ->orWhere('subcards.lastname', 'like', $keyword)
                    ->orWhere('subcards.company', 'like', $keyword)
                    ->orWhere('subcards.bio', 'like', $keyword)
                    ->orWhere('users.country', 'like', $keyword)
                    ->orWhere('users.state', 'like', $keyword)
                    ->orWhere('subcards.job_title', 'like', $keyword);
            });
        }

        // Merge main + sub queries
        $unionQuery = $queryMain->unionAll($querySub);

        // Paginate and return
        return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
            ->mergeBindings($unionQuery->getQuery())
            ->orderBy('order')
            ->paginate($this->perPage);
    }


    // public function getMembersProperty()
    // {
    //     $keyword = '%' . $this->keyword . '%';

    //     // Main card members
    //     $queryMain = Member::join('users', 'members.member_id', '=', 'users.id')
    //         ->leftJoin('user_settings as main_title', function ($join) {
    //             $join->on('main_title.user_id', '=', 'users.id')
    //                 ->where('main_title.setting_key', 'card_custom_title');
    //         })
    //         ->where('members.card_type', 'main')
    //         ->where('members.team_id', $this->team_id)
    //         ->select([
    //             'members.id as member_id',
    //             'members.member_id as entity_id',
    //             'members.card_type',
    //             'members.custom_card_link',
    //             'members.order',
    //             'users.username',
    //             'users.firstname',
    //             'users.lastname',
    //             'users.job_title',
    //             'users.company',
    //             'users.state',
    //             'users.bio',
    //             DB::raw('COALESCE(users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
    //             DB::raw('COALESCE(users.country, "") as country'),
    //             DB::raw('"main" as entity_type'),
    //             DB::raw('COALESCE(main_title.setting_value, CONCAT(users.firstname, " ", users.lastname)) as card_custom_title'),
    //         ]);

    //     // Sub card members
    //     $querySub = Member::join('subcards', 'members.member_id', '=', 'subcards.id')
    //         ->join('users', 'subcards.user_id', '=', 'users.id')
    //         ->leftJoin('user_settings as sub_title', function ($join) {
    //             $join->on('sub_title.page_id', '=', 'subcards.id')
    //                 ->where('sub_title.setting_key', 'card_custom_title');
    //         })
    //         ->where('members.card_type', 'sub')
    //         ->where('members.team_id', $this->team_id)
    //         ->select([
    //             'members.id as member_id',
    //             'members.member_id as entity_id',
    //             'members.card_type',
    //             'members.custom_card_link',
    //             'members.order',
    //             'subcards.username',
    //             'subcards.firstname',
    //             'subcards.lastname',
    //             'subcards.job_title',
    //             'subcards.company',
    //             'users.state',
    //             'subcards.bio',
    //             DB::raw('COALESCE(subcards.profile_pic, users.profile_pic, "/img/profile-placeholder.jpg") as profile_pic'),
    //             DB::raw('COALESCE(users.country, "") as country'),
    //             DB::raw('"sub" as entity_type'),
    //             DB::raw('COALESCE(sub_title.setting_value, CONCAT(subcards.firstname, " ", subcards.lastname)) as card_custom_title'),
    //         ]);

    //     // Apply search filter if keyword exists
    //     if ($this->keyword) {
    //         $queryMain->where(function ($q) use ($keyword) {
    //             $q->where('users.username', 'like', $keyword)
    //                 ->orWhere('users.firstname', 'like', $keyword)
    //                 ->orWhere('users.lastname', 'like', $keyword)
    //                 ->orWhere('users.bio', 'like', $keyword)
    //                 ->orWhere('users.company', 'like', $keyword)
    //                 ->orWhere('users.country', 'like', $keyword)
    //                 ->orWhere('users.state', 'like', $keyword)
    //                 ->orWhere('users.job_title', 'like', $keyword);
    //         });

    //         $querySub->where(function ($q) use ($keyword) {
    //             $q->where('subcards.username', 'like', $keyword)
    //                 ->orWhere('subcards.firstname', 'like', $keyword)
    //                 ->orWhere('subcards.lastname', 'like', $keyword)
    //                 ->orWhere('subcards.company', 'like', $keyword)
    //                 ->orWhere('subcards.bio', 'like', $keyword)
    //                 ->orWhere('users.country', 'like', $keyword)
    //                 ->orWhere('users.state', 'like', $keyword)
    //                 ->orWhere('subcards.job_title', 'like', $keyword);
    //         });
    //     }

    //     // Merge main + sub queries
    //     $unionQuery = $queryMain->unionAll($querySub);

    //     // Paginate
    //     return DB::table(DB::raw("({$unionQuery->toSql()}) as combined"))
    //         ->mergeBindings($unionQuery->getQuery())
    //         ->orderBy('order')
    //         ->paginate($this->perPage);
    // }

    public function render()
    {
        $this->thumbnail = $this->team->thumbnail ?: $this->user->profile_pic;
        $this->favicon = $this->user->thumbnail ?: $this->user->profile_pic;
        return view('livewire.display-team-page', [
            'members' => $this->members,
        ])
            ->title(($this->team->title ?? 'Team') . ' | ' . ($this->page_title ?: $this->card_title))
            ->layoutData([
                'favicon' => $this->thumbnail,
                'thumbnail' => $this->thumbnail,
                'username' => $this->username,
                'site_url' => $this->site_url,
                'title' => $this->page_title ?: $this->card_title,
                'color_settings_card_light' => $this->color_settings_card_light,
                'color_settings_card_dark' => $this->color_settings_card_dark,
                'color_settings_text_light' => $this->color_settings_text_light,
                'color_settings_text_dark' => $this->color_settings_text_dark,
                'app_name' => $this->app_name,
            ]);
    }
}
