<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Product;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AddSubscriptionProduct extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    // Product fields
    public $product_name,
        $price,
        $stock = 1,
        $coin,
        $shop_online_button = 'Subscribe',
        $shop_online_link,
        $description,
        $featured_image,
        $has_coins = false,
        $has_packages = false;

    public array $subscription_options = [];

    public $type = 'subscription';

    public $condition;

    public $page_id; // subcard id, used in query

    #[On('productImage')]
    public function setProductImage($productImageUrl)
    {
        $this->featured_image = new TemporaryUploadedFile($productImageUrl, config('filesystems.default'));
    }

    public function toggleHasPackages()
    {
        $this->has_packages = !$this->has_packages;
    }

    public function toggleHasCoins()
    {
        $this->has_coins = !$this->has_coins;
    }

    public function addProduct()
    {
        $this->subscription_options[] = [
            'title' => '',
            'price' => '',
            'coin' => '',
            'subscription_link' => '',
            'tier_description' => '',
        ];
    }

    public function removeProduct($index)
    {
        unset($this->subscription_options[$index]);
        $this->subscription_options = array_values($this->subscription_options); // reindex
    }

    public function saveProduct()
    {

        try {
            // Validation
            $validated = $this->validate([
                'featured_image'                            => 'required|image|max:2048',
                'product_name'                              => 'required|string',
                'price'                                     => 'nullable|numeric',
                'coin'                                      => 'nullable|numeric',
                'stock'                                     => 'nullable|numeric',
                'shop_online_button'                        => 'nullable|string',
                'shop_online_link'                          => 'nullable|url',
                'description'                               => 'required|string|max:1000',
                'subscription_options'                      => 'nullable|array',
                'subscription_options.*.title'              => 'nullable|string|max:255',
                'subscription_options.*.price'              => 'nullable|numeric',
                'subscription_options.*.coin'               => 'nullable|numeric',
                'subscription_options.*.subscription_link'  => 'nullable|url',
                'subscription_options.*.tier_description'   => 'nullable|string|max:5000',
                'type'                                      => 'required',
                'has_packages'                              => 'nullable|boolean',
                'has_coins'                                 => 'nullable|boolean',
            ]);

            // Handle featured image
            if ($this->featured_image) {
                $validated['featured_image'] = $this->featured_image->storePublicly('products', 'public');

                $tempFilePath = $this->featured_image->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath); // remove temporary file
                }
            }

            // Check active main card
            $checkifactive = User::where('id', auth()->user()->id)
                ->where('active_card', true)->count();

            if ($checkifactive < 1) {
                $this->page_id = SubCard::where('user_id', auth()->user()->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();
                $validated['page_id'] = $this->page_id;
            } else {
                $validated['user_id'] = auth()->user()->id;
            }

            $addProduct = Product::create($validated);

            if ($addProduct) {
                session()->flash('success', 'Subscription product successfully added!');
            } else {
                session()->flash('error', 'Unauthorized action.');
            }

            return $this->redirect('/dashboard/subscriptions', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function mount()
    {
        // $this->subscription_options[] = [
        //     'title' => '',
        //     'price' => '',
        //     'coin' => '',
        //     'subscription_link' => '',
        //     'tier_description' => '',
        // ];

        // Encode subscription_options as JSON if has_packages is true
        // if ($this->has_packages && !empty($this->subscription_options)) {
        //     $validated['subscription_options'] = $this->subscription_options ?? [];
        //     $validated['price'] = null;
        //     $validated['coin'] = null;
        //     $validated['shop_online_link'] = null;
        // } else {
        //     $validated['subscription_options'] = null;
        // }

        $checkIfActive = User::where('id', auth()->id())
            ->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())
                ->where('active_card', true)
                ->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
    }

    public function render()
    {
        $this->authorize('create', Product::class);
        return view('livewire.add-subscription-product')
            ->title('Add Subscription Product | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
