<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Sale;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;

class AddSalesPage extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;

    public $title, $description, $image;
    public $is_active = true;

    public $page_id; //subcard id, used in query

    public function toggleActive()
    {
        $this->is_active = !$this->is_active;
    }

    #[On('salesImage')]
    public function setSalesImage($eventImageUrl)
    {
        $this->image = new TemporaryUploadedFile($eventImageUrl, config('filesystems.default'));
    }

    public function savePage()
    {
        $validated = $this->validate([
            'image'         => 'required|image|max:2048',
            'title'         => 'required|string',
            'description'   => 'nullable|string',
            'is_active'     => 'required|boolean',
        ]);

        $checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        //if maincard is not active
        if ($checkifactive < 1) {
            $this->page_id = SubCard::where('user_id', auth()->user()->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();

            $validated['page_id'] = $this->page_id;
        } else {
            $validated['user_id'] = auth()->user()->id;
        }

        // Handle image upload
        if ($this->image instanceof TemporaryUploadedFile) {
            $validated['thumbnail'] = $this->image->storePublicly('sales', 'public');

            $tempFilePath = $this->image->getRealPath();
            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);
            }
        }

        $addSalesPage = Sale::create($validated);

        if ($addSalesPage) {
            session()->flash('success', 'Sales Page successfully added!');
            return $this->redirectRoute('sales.edit', [
                'id' => $addSalesPage->id,
            ], navigate: true);
        } else {
            session()->flash('error', 'Unauthorized action.');
        }
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;

        $this->authorize('create', Sale::class);
        return view('livewire.add-sales-page')->title('Add Sales Page | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
        ]);
    }
}
