<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Product;
use App\Models\SubCard;
use App\Models\User;
use App\Models\ProductCategory;
use App\Models\ProductCollection;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AddProduct extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    // Product fields
    public $product_name, $price, $stock = 1, $shop_online_button = 'Shop Online', $shop_online_link, $description, $featured_image;

    // New fields
    public $category_id = [];
    public $collection_id = [];
    public $type = 'physical';
    public $sizesInput = '';
    public $stylesInput = '';
    public $colors = [];
    public $categories;
    public $collections;

    public $condition;

    public $page_id; // subcard id, used in query

    #[On('productImage')]
    public function setProductImage($productImageUrl)
    {
        $this->featured_image = new TemporaryUploadedFile($productImageUrl, config('filesystems.default'));
    }

    // Add a new color entry
    public function addColor()
    {
        $this->colors[] = ['name' => '', 'value' => '#ffffff'];
    }

    // Remove color by index
    public function removeColor($index)
    {
        unset($this->colors[$index]);
        $this->colors = array_values($this->colors);
    }

    public function saveProduct()
    {
        try {
            // Validation
            $validated = $this->validate([
                'featured_image'        => 'required|image|max:2048',
                'product_name'          => 'required|string',
                'price'                 => 'required|numeric',
                'stock'                 => 'required|numeric',
                'shop_online_button'    => 'nullable|string',
                'shop_online_link'      => 'required|url',
                'description'           => 'required|string|max:5000',
                // 'collection_id'         => 'nullable|exists:products_collection,id',
                // 'category_id'           => 'nullable|exists:product_categories,id',
                'category_id'   => 'nullable|array',
                'category_id.*' => 'exists:product_categories,id',

                'collection_id'   => 'nullable|array',
                'collection_id.*' => 'exists:products_collection,id',
                'type'                  => 'required|in:physical,digital',
                'sizesInput'            => $this->type !== 'digital' ? 'nullable|string' : 'nullable',
                'stylesInput'           => $this->type !== 'digital' ? 'nullable|string' : 'nullable',
                'colors.*.name'         => $this->type !== 'digital' ? 'required|string' : 'nullable',
                'colors.*.value'        => $this->type !== 'digital' ? 'required|string' : 'nullable',
            ]);

            $validated['category_id'] = empty($this->category_id)
                ? null
                : array_map('intval', $this->category_id);

            $validated['collection_id'] = empty($this->collection_id)
                ? null
                : array_map('intval', $this->collection_id);

            // Convert comma-separated sizes and styles into arrays
            // $sizes = array_map('trim', explode(',', $this->sizesInput));
            // $styles = array_map('trim', explode(',', $this->stylesInput));

            if ($this->type === 'physical') {

                // Sizes (ALWAYS clean empty values)
                $sizes = array_values(array_filter(
                    is_array($this->sizesInput)
                        ? $this->sizesInput
                        : array_map('trim', explode(',', (string) $this->sizesInput)),
                    fn($value) => trim($value) !== ''
                ));

                // Styles (ALWAYS clean empty values)
                $styles = array_values(array_filter(
                    is_array($this->stylesInput)
                        ? $this->stylesInput
                        : array_map('trim', explode(',', (string) $this->stylesInput)),
                    fn($value) => trim($value) !== ''
                ));

                $validated['sizes']  = json_encode($sizes);
                $validated['styles'] = json_encode($styles);
                $validated['colors'] = json_encode($this->colors);
            } else {
                $validated['sizes']  = json_encode([]);
                $validated['styles'] = json_encode([]);
                $validated['colors'] = null;
            }

            // Handle featured image
            if ($this->featured_image) {
                $validated['featured_image'] = $this->featured_image->storePublicly('products', 'public');

                $tempFilePath = $this->featured_image->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath); // remove temporary file
                }
            }

            // Check active main card
            $checkifactive = User::where('id', auth()->user()->id)
                ->where('active_card', true)->count();

            if ($checkifactive < 1) {
                $this->page_id = SubCard::where('user_id', auth()->user()->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();
                $validated['page_id'] = $this->page_id;
            } else {
                $validated['user_id'] = auth()->user()->id;
            }

            try {
                // Save product
                $addProduct = Product::create(array_merge($validated, [
                    'sizes' => $sizes,
                    'styles' => $styles,
                    'colors' => $this->colors,
                ]));

                if ($addProduct) {
                    session()->flash('success', 'Product successfully added!');
                } else {
                    session()->flash('error', 'Unauthorized action.');
                }
            } catch (\Throwable $e) {
                \Log::error("PRODUCT ERROR: " . $e->getMessage());
                \Log::error($e->getTraceAsString());
                throw $e; // show error directly in browser
            }

            return $this->redirect('/dashboard/product', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function mount()
    {
        $checkIfActive = User::where('id', auth()->id())
            ->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())
                ->where('active_card', true)
                ->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        // Load categories for dropdown
        $this->categories = ProductCategory::where($this->condition)->get();
        $this->collections = ProductCollection::where($this->condition)->get();
    }

    public function render()
    {
        $this->authorize('create', Product::class);
        return view('pages.add-product')
            ->title('Add Product | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
