<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\PresentationPage;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AddPresentationPage extends Component
{
    use AuthorizesRequests, WithFileUploads;

    public $page_id; // for editing or linking to subcard
    public $title;
    public $description;
    public $is_active = true;
    public $contents = []; // blocks

    // Site settings
    public $site_name;
    public $thumbnail;
    public $app_name;

    public function mount($page_id = null)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        // Load existing page for editing
        if ($page_id) {
            $this->page_id = $page_id;
            $page = PresentationPage::find($page_id);
            if ($page) {
                $this->title = $page->title;
                $this->description = $page->description;
                $this->is_active = $page->is_active;
                // decode contents JSON safely
                $this->contents = is_array($page->contents) ? $page->contents : json_decode($page->contents, true) ?? [];
                // ensure each block has 'file' key for Livewire uploads
                foreach ($this->contents as &$block) {
                    if (!isset($block['file'])) $block['file'] = null;
                }
            }
        }
    }

    public function addBlock(string $type)
    {
        $this->contents[] = [
            'type' => $type,
            'title' => null,
            'value' => null,
            'file' => null,
        ];
    }

    public function updateOrder(array $order)
    {
        $newContents = [];
        foreach ($order as $i) {
            $newContents[] = $this->contents[$i];
        }
        $this->contents = $newContents;
    }

    public function removeBlock(int $index)
    {
        if (isset($this->contents[$index])) {
            unset($this->contents[$index]);
            $this->contents = array_values($this->contents);
        }
    }

    public function moveUp(int $index)
    {
        if ($index <= 0) return;
        [$this->contents[$index - 1], $this->contents[$index]] = [$this->contents[$index], $this->contents[$index - 1]];
    }

    public function moveDown(int $index)
    {
        if ($index >= count($this->contents) - 1) return;
        [$this->contents[$index + 1], $this->contents[$index]] = [$this->contents[$index], $this->contents[$index + 1]];
    }

    protected function rules(): array
    {
        return [
            'title'                         => 'required|string|max:255',
            'description'                   => 'nullable|string|max:200',
            'is_active'                     => 'required|boolean',
            'contents'                      => 'nullable|array',
            'contents.*.type'               => 'required|string',
            'contents.*.title'              => 'nullable|string',
            'contents.*.value'              => 'nullable|string',
            'contents.*.file'               => 'nullable|image|max:2048',
            'contents.*.call_to_action'     => 'nullable|url',
        ];
    }

    public function savePage()
    {
        try {
            // Ensure contents is always an array
            if (is_string($this->contents)) {
                $decoded = json_decode($this->contents, true);
                $this->contents = is_array($decoded) ? $decoded : [];
            }

            $this->validate();

            $processed = [];

            foreach ($this->contents as $i => $block) {
                $type = $block['type'] ?? null;
                if (!$type) continue;

                switch ($type) {
                    case 'heading':
                    case 'html':
                    case 'video':
                        $value = trim($block['value'] ?? '');

                        if ($value === '') {
                            $this->addError("contents.$i.value", ucfirst($type) . " block cannot be empty.");
                            return;
                        }
                        if ($type === 'video') {
                            if (!filter_var($block['value'], FILTER_VALIDATE_URL)) {
                                $this->addError("contents.$i.value", "Please enter a valid URL.");
                                return;
                            }
                        }

                        $processed[] = [
                            'type' => $type,
                            'title' => $block['title'] ?? null,
                            'value' => $value,
                        ];
                        break;

                    case 'button':
                        $label = trim($block['title'] ?? '');
                        $url = trim($block['value'] ?? '');

                        // Initialize error flag
                        $hasError = false;

                        // 1. Required fields
                        if ($label === '') {
                            // session()->flash('error', "Button block (#" . ($i + 1) . ") requires a label and URL.");
                            $this->addError("contents.$i.title", "Button block requires a label.");
                            $hasError = true;
                        }

                        if ($url === '') {
                            // session()->flash('error', "Button block (#" . ($i + 1) . ") requires a label and URL.");
                            $this->addError("contents.$i.value", "Button block requires a URL.");
                            $hasError = true;
                        }

                        // 2. Check if valid URL
                        if (!filter_var($url, FILTER_VALIDATE_URL)) {
                            // session()->flash('error', "Button block (#" . ($i + 1) . ") must contain a valid URL.");
                            $this->addError("contents.$i.value", "Please enter a valid URL.");
                            $hasError = true;
                        }

                        // If any errors exist, stop processing
                        if ($hasError) {
                            return;
                        }

                        $processed[] = [
                            'type' => 'button',
                            'title' => $label,
                            'value' => $url,
                        ];
                        break;

                    case 'image':
                        $uploaded = $block['file'] ?? null;
                        $existing = $block['value'] ?? null;
                        $call_to_action = $block['call_to_action'] ?? null;

                        if ($uploaded instanceof TemporaryUploadedFile) {
                            $path = $uploaded->storePublicly('pages', 'public');
                            $processed[] = [
                                'type' => 'image',
                                'title' => null,
                                'value' => $path,
                                'call_to_action' => $call_to_action,
                            ];
                        } elseif (!empty($existing)) {
                            $processed[] = [
                                'type' => 'image',
                                'title' => null,
                                'value' => $existing,
                                'call_to_action' => $call_to_action,
                            ];
                        } else {
                            // session()->flash('error', "Image block (#" . ($i + 1) . ") requires an uploaded file.");
                            $this->addError("contents.$i.file", "Image block requires an uploaded file.");
                            return;
                        }

                        break;

                    default:
                        break;
                }
            }

            $checkIfActive = User::where('id', auth()->id())->where('active_card', true)->exists();

            $data = [
                'title' => $this->title,
                'description' => $this->description,
                'is_active' => $this->is_active,
                // 'contents' => json_encode($processed),
                'contents' => $processed,
                'user_id' => $checkIfActive ? auth()->id() : null,
            ];

            if (!$checkIfActive) {
                $this->page_id = SubCard::where('user_id', auth()->id())->where('active_card', true)->value('id');
                $data['page_id'] = $this->page_id;
            }

            // dd($data);

            try {
                PresentationPage::create($data);

                session()->flash('success', 'Page successfully saved!');
                return $this->redirect('/dashboard/pages', navigate: true);
            } catch (\Throwable $e) {
                // \Log::error('Failed to save presentation page: ' . $e->getMessage());
                session()->flash('error', 'Something went wrong while saving the page.');
                return redirect()->back()->withInput();
            }
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        return view('livewire.add-presentation-page')
            ->title('Add Page | ' . $this->site_name)
            ->layoutData(['thumbnail' => $this->thumbnail, 'app_name' => $this->app_name,]);
    }
}
