<?php

namespace App\Livewire;

use App\Models\Button;
use Livewire\Component;
use App\Models\Sale;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class AddNewTeam extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $title, $description, $image;
    public $is_active = true;

    public $page_id; //subcard id, used in query
    public $team_thumbnail;

    public function toggleActive()
    {
        $this->is_active = !$this->is_active;
    }

    #[On('galleryImage')]
    public function setImage($newImageUrl)
    {
        $this->team_thumbnail = new TemporaryUploadedFile($newImageUrl, config('filesystems.default'));
    }

    public function saveTeam()
    {
        try {
            $validated = $this->validate([
                // 'image'         => 'required|image|max:2048',
                'title'         => 'required|string',
                'description'   => 'nullable|string|max:1000',
                'is_active'     => 'required|boolean',
            ]);

            $checkifactive = User::where('id', auth()->user()->id)
                ->where('active_card', true)->count();

            $validated['type'] = 'team';
            $validated['view'] = 'links';
            $validated['user_id'] = auth()->user()->id;

            //if maincard is not active
            if ($checkifactive < 1) {
                $this->page_id = SubCard::where('user_id', auth()->user()->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();

                $validated['page_id'] = $this->page_id;
            }

            // dd($validated['user_id'], $validated['page_id']);

            if ($this->team_thumbnail) {
                // Store the new image
                $validated['thumbnail'] = $this->team_thumbnail->storePublicly('thumbnails', 'public');

                // Delete temp file if still there (optional, Laravel usually cleans this up)
                $tempFilePath = $this->team_thumbnail->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);
                }
            }

            $addNewTeam = Button::create($validated);

            if ($addNewTeam) {
                session()->flash('success', 'New Team successfully added!');
                return $this->redirectRoute('team.edit', [
                    'id' => $addNewTeam->id,
                ], navigate: true);
            } else {
                session()->flash('error', 'Unauthorized action.');
            }
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->authorize('create', Button::class);
        return view('livewire.add-new-team')->title('Add New Team | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
