<?php

namespace App\Livewire;

use Livewire\Component;
use App\Settings\CardDefaultsSetting;
use App\Helpers\SavingLinkHelpers;
use App\Models\Button;
use App\Models\SubCard;
use App\Models\UserSettings;
use Livewire\Attributes\On;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class AddNewPage extends Component
{
    use AuthorizesRequests;

    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $user;
    public $firstname, $lastname;
    public $username;
    public $email;
    public $job_title;
    public $company;
    public $company_website;
    public $profile_pic;
    public $bio;
    public $bio_long;
    public $phone;

    #[On('profilePic')]
    public function setProfilePic($profileImageUrl)
    {
        $this->profile_pic = new TemporaryUploadedFile($profileImageUrl, config('filesystems.default'));
    }

    public function addNewSubCard()
    {
        if (auth()->user()) {

            $validated = $this->validate([
                // 'username' => 'required|string|max:50|unique:users|unique:subcards',
                'username' => [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_.]+$/',
                    'unique:users',
                    'unique:subcards',
                ],
                'firstname' => 'required|string|max:50',
                'lastname' => 'required|string|max:50',
                // 'email' => 'required|string|unique:users|unique:subcards',
                'email' => 'nullable|string',
                'phone' => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
                'job_title' => 'required|string|max:200',
                'bio' => 'nullable|string|max:200',
                'bio_long' => 'nullable|string|max:1000',
                'company' => 'nullable|string|max:255',
                'company_website' => 'nullable|url|max:255',
            ], [
                'username.regex' => "Username may only contain letters, numbers, and underscores.",
                'phone.regex' => 'Please enter a valid phone number.',
            ]);

            $validated['user_id'] = auth()->user()->id;
            $validated['is_active'] = true;

            if ($this->company_website) {
                try {
                    // will validate and make the company website secured via https
                    $securedLink = SavingLinkHelpers::saveSecureLink($this->company_website);
                    $this->company_website = $securedLink;
                } catch (\InvalidArgumentException $e) {
                    session()->flash('company_website', 'Invalid company website. Please enter a valid company website.');
                    // return $this->redirect('/dashboard/card/add', navigate: true);
                    return redirect()->back();
                }
            }

            if ($this->profile_pic) {
                $validated['profile_pic'] =  $this->profile_pic->storePublicly('profile_pic', 'public');

                // Get the path of the temporary file (before it's stored in the public directory)
                $tempFilePath = $this->profile_pic->getRealPath();

                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);  // Delete the temporary file explicitly
                }
            }

            // Update the user data
            $create = auth()->user()->subcards()->create($validated);

            if ($create) {
                session()->flash('success', 'Subcard successfully created.');
                return $this->redirect('/dashboard/profile', navigate: true);
            }
        } else {
            session()->flash('error', 'Unauthorized action.');
            return $this->redirect('/dashboard/profile', navigate: true);
        }
    }

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_url = $default->site_url;
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
    }

    public function render()
    {
        $this->authorize('create', SubCard::class);
        return view('pages.add-new-page')
            ->title('Add New Page | ' . $this->site_name)->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
