<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Ad;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AddAd extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $ad_banner_w;
    public $ad_banner_h;
    public $app_name;

    public $link, $caption, $image, $position, $is_active;

    public $page_id; //subcard id, used in query

    #[On('adImage')]
    public function setAdImage($adImageUrl)
    {
        //dd($productImageUrl);
        $this->image = new TemporaryUploadedFile($adImageUrl, config('filesystems.default'));
        // dd($this->product_image);
    }

    public function saveAd()
    {
        try {
            $validated = $this->validate(
                [
                    'image' => 'required|image|mimes:jpeg,png,jpg,webp,heic,heif|max:5120',
                    'caption' => 'required|string',
                    'link' => 'required|string',
                    'position' => 'required|in:top,bottom',
                ], //Custom error messages
                [
                    'caption.required' => "Ad Title is required.",
                    'image.required' => "Ad Banner is required.",
                    'image.image' => "The file must be an image (jpeg, png, jpg, heic, heif or webp).",
                ]
            );

            $validated['user_id'] = auth()->user()->id;

            if ($this->image) {
                $validated['image'] =  $this->image->storePublicly('ads', 'public');

                // Get the path of the temporary file (before it's stored in the public directory)
                $tempFilePath = $this->image->getRealPath();

                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);  // Delete the temporary file explicitly
                }
            }

            $checkifactive = User::where('id', auth()->user()->id)
                ->where('active_card', true)->count();

            //if maincard is not active
            if ($checkifactive < 1) {
                $this->page_id = SubCard::where('user_id', auth()->user()->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();
                $validated['page_id'] = $this->page_id;
            }

            $saveTheAd = Ad::create($validated);

            if ($saveTheAd) {
                session()->flash('success', 'Ad successfully added!');
            } else {
                session()->flash('error', 'Unauthorized action.');
            }

            return $this->redirect('/dashboard/ads', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        $this->ad_banner_w = $default->ad_banner_w;
        $this->ad_banner_h = $default->ad_banner_h;

        $this->authorize('create', Ad::class);
        return view('livewire.add-ad')->title('Add Ad | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
