<?php

namespace App\Listeners;

use App\Models\User;
use Laravel\Cashier\Events\WebhookReceived;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Subscription as StripeSubscription;

class SyncUserSubscriptionType
{
    public function handle(WebhookReceived $event): void
    {
        $payload   = $event->payload;
        $eventType = $payload['type'] ?? 'unknown';

        // Only process important events
        $processableEvents = [
            'customer.subscription.created',
            'customer.subscription.updated',
            'invoice.payment_succeeded',
            'invoice.paid',
        ];

        if (!in_array($eventType, $processableEvents)) {
            Log::info('[Stripe Webhook] Ignored event type: ' . $eventType);
            return;
        }

        $object           = $payload['data']['object'] ?? [];
        $stripeCustomerId = $object['customer'] ?? null;

        if (!$stripeCustomerId) {
            Log::warning('[Stripe Webhook] Missing customer ID');
            return;
        }

        $user = \Laravel\Cashier\Cashier::findBillable($stripeCustomerId);
        if (!$user) {
            Log::warning('[Stripe Webhook] No user found for Stripe customer', [
                'customer_id' => $stripeCustomerId,
            ]);
            return;
        }

        Stripe::setApiKey(config('cashier.secret') ?? config('services.stripe.secret'));

        $subscription = $user->subscription('default');
        $stripeSub    = null;

        /**
         * ----------------------------------------------------------
         * STEP 1: Fetch Stripe subscription and sync local fields
         * ----------------------------------------------------------
         */
        try {
            if ($subscription && $subscription->stripe_id) {
                $stripeSub = StripeSubscription::retrieve($subscription->stripe_id);

                $subscription->update([
                    'stripe_status' => $stripeSub->status,
                    'ends_at'        => $stripeSub->cancel_at ? now() : null,
                ]);
            }
        } catch (\Exception $e) {
            Log::error('[Stripe Webhook] Error fetching Stripe subscription', [
                'user_id' => $user->id,
                'error'   => $e->getMessage(),
            ]);
        }

        $status = $subscription?->stripe_status ?? 'none';

        /**
         * ----------------------------------------------------------
         * STEP 2: List ALL Stripe subscriptions to detect real state
         * ----------------------------------------------------------
         */
        try {
            $stripeSubs = StripeSubscription::all([
                'customer' => $stripeCustomerId,
                'status'   => 'all',
                'limit'    => 10,
            ]);
        } catch (\Exception $e) {
            Log::error('[Stripe Webhook] Error checking Stripe subscriptions', [
                'user_id' => $user->id,
                'error'   => $e->getMessage(),
            ]);
            return;
        }

        // FIX: Correct active detection (includes trial)
        $hasActiveOrTrialing = collect($stripeSubs->data)->contains(function ($s) {
            return in_array($s->status, [
                'active',
                'trialing',
                'past_due',
                'incomplete',
            ]);
        });

        /**
         * ----------------------------------------------------------
         * STEP 3: Skip downgrade during payment or creation flow
         * ----------------------------------------------------------
         */
        $skipDowngradeEvents = [
            'invoice.payment_succeeded',
            'invoice.paid',
            'customer.subscription.created',
            'invoice.updated',
            'invoice.finalized',
            'payment_intent.succeeded',
        ];

        $skipDowngrade = in_array($eventType, $skipDowngradeEvents);

        // FIX: Only downgrade when it's NOT a payment/upgrade AND no valid subscriptions
        if (!$skipDowngrade && !$hasActiveOrTrialing) {

            $user->update([
                'user_type'     => 'Free',
                'access'        => null,
                'trial_ends_at' => null,
            ]);

            if ($subscription && $subscription->stripe_status !== 'canceled') {
                $subscription->cancelNow();
            }

            Log::info('[Stripe Webhook] User downgraded to Free', [
                'user_id' => $user->id,
            ]);

            return;
        }

        /**
         * ----------------------------------------------------------
         * STEP 4: Determine plan (from metadata or price ID)
         * ----------------------------------------------------------
         */
        $metadata = $object['metadata'] ?? [];
        $plan     = $metadata['plan'] ?? null;

        if (!$plan && $stripeSub) {
            $priceId = $stripeSub->items->data[0]->price->id ?? null;

            $default = app(\App\Settings\CardDefaultsSetting::class);
            $isTest  = $default->test_mode;

            $plans = [
                $isTest ? $default->test_basic_yearly    : $default->live_basic_yearly    => 'basic',
                $isTest ? $default->test_basic_monthly   : $default->live_basic_monthly   => 'basic',

                $isTest ? $default->test_gold_yearly     : $default->live_gold_yearly     => 'gold',
                $isTest ? $default->test_gold_monthly    : $default->live_gold_monthly    => 'gold',

                $isTest ? $default->test_platinum_yearly : $default->live_platinum_yearly => 'platinum',
                $isTest ? $default->test_platinum_monthly : $default->live_platinum_monthly => 'platinum',
            ];

            $plan = $plans[$priceId] ?? null;
        }

        /**
         * ----------------------------------------------------------
         * STEP 5: Payment success → activate subscription properly
         * ----------------------------------------------------------
         */
        if (in_array($eventType, ['invoice.payment_succeeded', 'invoice.paid'])) {

            // FIX: Mark subscription active if Stripe has activated it
            if ($subscription && in_array($status, ['incomplete', 'past_due', 'unpaid'])) {
                $subscription->update(['stripe_status' => 'active']);
            }

            // FIX: End user's trial ONLY after successful payment
            if ($user->trial_ends_at) {
                $user->update(['trial_ends_at' => null]);
            }

            if ($subscription && $subscription->trial_ends_at) {
                $subscription->update(['trial_ends_at' => null]);
            }

            // FIX: Set user plan
            if ($plan) {
                $user->update([
                    'user_type'     => ucfirst($plan),
                    'access'        => null,
                    'trial_ends_at' => null,
                ]);

                Log::info('[Stripe Webhook] Upgraded user after payment success', [
                    'user_id' => $user->id,
                    'plan'    => ucfirst($plan),
                ]);
            }

            return;
        }

        /**
         * ----------------------------------------------------------
         * STEP 6: On trial → keep user Free (do NOT cancel trial)
         * ----------------------------------------------------------
         */
        if ($status === 'trialing') {
            $user->update(['user_type' => 'Free']);

            Log::info('[Stripe Webhook] Subscription trialing → user stays Free', [
                'user_id' => $user->id,
            ]);
        }
    }
}
