<?php

namespace App\Listeners;

use Laravel\Cashier\Events\WebhookReceived;
use Illuminate\Support\Facades\Log;
use App\Models\Commissions;
use App\Models\User;
use App\Models\Referral;
use App\Models\SubCard;

class HandleCommissionAfterInvoice
{
    public function handle(WebhookReceived $event): void
    {
        if ($event->payload['type'] !== 'invoice.payment_succeeded') {
            return;
        }

        Log::info('Processing invoice.payment_succeeded');

        $stripeInvoice = $event->payload['data']['object'];

        // Prevent duplicate commission entries
        if (Commissions::where('stripe_invoice_id', $stripeInvoice['id'])->exists()) {
            Log::info('Commission already recorded for invoice: ' . $stripeInvoice['id']);
            return;
        }

        $user = User::where('stripe_id', $stripeInvoice['customer'])->first();

        if (!$user) {
            Log::warning('User not found for Stripe customer ID: ' . $stripeInvoice['customer']);
            return;
        }

        $refer = Referral::where('referred_id', $user->id)->first();

        if (!$refer) {
            Log::info('No referral record found for user: ' . $user->id);
            return;
        }

        if ($refer->referrer_type == 'main') {
            $referrerId = $refer->referrer_id_main;
            $referrer = User::find($referrerId);
            $partnershipLevel = $referrer?->partnership_level;
        } else {
            $referrerId = $refer->referrer_id_sub;
            $subcard = SubCard::find($referrerId);

            if (!$subcard) {
                Log::warning('SubCard not found: ' . $referrerId);
                return;
            }

            $referrer = User::find($subcard->user_id);
            $partnershipLevel = $referrer?->partnership_level;
        }

        if (!$referrerId || !$referrer) {
            Log::warning('Referrer not found for user: ' . $user->id);
            return;
        }

        Log::info("Referrer partnership level: " . $partnershipLevel);

        $rate = match ($partnershipLevel) {
            'senior' => 0.50,
            'junior' => 0.25,
            'leader' => 0.20,
            default => 0.10,
        };

        $total = $stripeInvoice['amount_paid'] / 100;

        $commissionData = [
            'referred_id' => $refer->referred_id,
            'stripe_invoice_id' => $stripeInvoice['id'],
            'commission_rate' => $rate * 100,
            'amount' => $total * $rate,
        ];

        if ($refer->referrer_type === 'main') {
            $commissionData['user_id'] = $referrerId;
            $commissionData['page_id'] = null;
            Log::info("Main: {$referrerId}");
        } else {
            $commissionData['user_id'] = null;
            $commissionData['page_id'] = $referrerId;
            Log::info("Sub: {$referrerId}");
        }

        Commissions::create($commissionData);

        Log::info("Commission created for referrer {$referrerId}, amount: " . $total * $rate);
    }
}
