<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Carbon\Carbon;
use Malzariey\FilamentLexicalEditor\Enums\ToolbarItem;
use Malzariey\FilamentLexicalEditor\FilamentLexicalEditor;
use Illuminate\Support\Str;
use Filament\Forms\Components\FileUpload;
use Illuminate\Database\Eloquent\Model;
use App\Settings\CardDefaultsSetting;
use Kahusoftware\FilamentCkeditorField\CKEditor;

class PagesRelationManager extends RelationManager
{
    protected static string $relationship = 'pages';
    protected static ?string $modelLabel = 'Page';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return in_array($ownerRecord->user_type, ['Gold', 'Platinum']);
    }

    protected array $types = [
        'button' => [
            'label' => 'Button',
            'field_label' => 'Link',
        ],
        'heading' => [
            'label' => 'Text Heading',
            'field_label' => 'Content',
        ],
        'html' => [
            'label' => 'Text Content',
            'field_label' => 'Content',
        ],
        'video' => [
            'label' => 'Video',
            'field_label' => 'Youtube, Vimeo, Instagram or Tiktok',
        ],
        'image' => [
            'label' => 'Image',
            'field_label' => 'Upload Image',
            'upload_dir' => 'pages',
        ],
    ];

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('title')
                    ->label('Page Title')
                    ->maxLength(300)
                    ->placeholder('ex: Presentation Page, Main Menu')
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\Textarea::make('description')
                    ->label('Description')
                    ->maxLength(1000)
                    ->columnSpanFull(),

                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\Repeater::make('contents')
                    ->label('Contents')
                    ->itemLabel(function ($state) {
                        return $state['type'] ?? 'Content Block';
                    })
                    ->collapsible()
                    ->columnSpanFull()
                    ->schema([
                        // Type selector
                        Forms\Components\Select::make('type')
                            ->label('Element Type')
                            ->required()
                            ->reactive()
                            // ->options(array_map(fn($item) => $item['label'], $this->types))
                            ->options(
                                collect($this->types)->mapWithKeys(
                                    fn($item, $key) => [$key => $item['label']]
                                )->toArray()
                            )
                            ->default(fn($record) => $record?->type),

                        Forms\Components\TextInput::make('title')
                            ->label('Title')
                            ->maxLength(255)
                            ->required(fn(callable $get) => $get('type') === 'button')
                            ->visible(fn(callable $get, $livewire) => $get('type') === 'button')
                            ->afterStateHydrated(function ($component, $state, $record) {
                                if ($record && $record->type === 'button') {
                                    $component->state($record->title);
                                }
                            }),

                        // TextInput for button and video
                        Forms\Components\TextInput::make('value')
                            ->label(fn(callable $get) => match ($get('type')) {
                                'button' => 'Link',
                                'heading' => 'Content',
                                'video' => 'Youtube, Vimeo, Instagram or Tiktok',
                                default => 'Value',
                            })
                            ->url(fn(callable $get) => in_array($get('type'), ['button', 'video']))
                            ->required(fn(callable $get) => in_array($get('type'), ['heading', 'button', 'video']))
                            ->visible(fn(callable $get, $livewire) => in_array($get('type'), ['heading', 'button', 'video']))
                            ->afterStateHydrated(function ($component, $state, $record) {
                                if ($record && in_array($record->type, ['heading', 'button', 'video'])) {
                                    $component->state($record->value);
                                    logger($record->value);
                                }
                            }),

                        // CKEditor::make('value')
                        //     ->columnSpanFull()
                        //     // ->reactive()
                        //     // ->live()
                        //     ->label('Text Content')
                        //     ->extraAttributes(['wire:ignore'])
                        //     ->required(fn(callable $get) => $get('type') === 'html')
                        //     ->visible(fn($get) => $get('type') === 'html')

                        //     ->afterStateHydrated(function ($component, $state, $record) {
                        //         if ($record && $record->type === 'html') {
                        //             $component->state($record->value);
                        //         }
                        //     }),

                        Forms\Components\RichEditor::make('value')
                            ->columnSpanFull()
                            ->disableGrammarly()
                            ->label('Text Content')
                            ->toolbarButtons([
                                // 'blockquote',
                                'bold',
                                'bulletList',
                                // 'h1',
                                // 'h2',
                                // 'h3',
                                'italic',
                                'link',
                                'orderedList',
                                // 'redo',
                                // 'underline',
                                // 'undo',
                            ])
                            ->required(fn(callable $get) => $get('type') === 'html')
                            ->visible(fn($get) => $get('type') === 'html')

                            ->afterStateHydrated(function ($component, $state, $record) {
                                if ($record && $record->type === 'html') {
                                    $component->state($record->value);
                                }
                            }),

                        // FilamentLexicalEditor::make('value')
                        //     ->reactive()
                        //     ->live()
                        //     ->label('Text Content')
                        //     ->enabledToolbars([
                        //         ToolbarItem::BOLD,
                        //         ToolbarItem::ITALIC,
                        //         ToolbarItem::UNDERLINE,
                        //         ToolbarItem::LINK,
                        //         ToolbarItem::LEFT,
                        //         ToolbarItem::CENTER,
                        //         ToolbarItem::RIGHT,
                        //         ToolbarItem::JUSTIFY,
                        //         ToolbarItem::START,
                        //         ToolbarItem::END,
                        //         ToolbarItem::BULLET,
                        //         ToolbarItem::NUMBERED,
                        //     ])
                        //     ->required(fn(callable $get) => $get('type') === 'html')
                        //     // ->hidden(fn(callable $get, $livewire) => $get('type') != 'html')
                        //     ->visible(fn($get) => $get('type') === 'html')

                        //     ->afterStateHydrated(function ($component, $state, $record) {
                        //         if ($record && $record->type === 'html') {
                        //             $component->state($record->value);
                        //         }
                        //     }),

                        FileUpload::make('value')
                            ->label('Upload Image')
                            ->directory('pages')
                            ->image()
                            ->multiple(false)
                            ->imageEditor()
                            ->imageResizeMode('cover')
                            ->imageResizeTargetWidth(720)
                            ->imageResizeTargetHeight(720)
                            ->required(fn($get) => $get('type') === 'image')
                            ->visible(fn($get) => $get('type') === 'image')
                            ->afterStateHydrated(function ($component, $state, $record) {

                                if (!empty($record['contents'])) {
                                    foreach ($record['contents'] as $item) {
                                        if (($item['type'] ?? null) === 'image' && ($item['value'] ?? null) === $state) {
                                            // Only wrap the state for image fields
                                            if ($state && is_string($state)) {
                                                $component->state([$state]);
                                            }
                                            break;
                                        }
                                    }
                                }
                            })

                            ->getUploadedFileNameForStorageUsing(fn($file) => Str::random(40) . '.' . $file->getClientOriginalExtension()),

                        Forms\Components\TextInput::make('call_to_action')
                            ->label('Call to Action')
                            ->url()
                            // ->required(fn(callable $get) => $get('type') === 'image')
                            ->visible(fn(callable $get, $livewire) => $get('type') === 'image')
                            ->afterStateHydrated(function ($component, $state, $record) {
                                if ($record && $record->type === 'image') {
                                    $component->state($record->call_to_action);
                                    logger($record->value);
                                }
                            }),
                    ]),
            ]);
    }

    public static function getScripts(): array
    {
        return [
            <<<JS
document.addEventListener('livewire:load', () => {

    const editors = {};

    const initCKEditors = () => {
        document.querySelectorAll('[data-ckeditor-field]').forEach((el) => {

            const id = el.getAttribute('id');

            if (!id) return;

            // If exists, destroy old instance before creating new
            if (editors[id]) {
                editors[id].destroy()
                    .then(() => delete editors[id])
                    .catch(() => {});
            }

            if (window.filamentCKEditorInit) {
                window.filamentCKEditorInit(el).then(editor => {
                    editors[id] = editor;
                });
            }
        });
    };

    // Run on initial load
    setTimeout(initCKEditors, 300);

    // Run after any Livewire DOM update
    Livewire.hook('message.processed', () => {
        setTimeout(initCKEditors, 150);
    });

});
JS
        ];
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->orderBy('order', 'desc')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive
                Tables\Columns\TextColumn::make('title'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => $state ? Carbon::parse($state)->format('M d, Y') : null),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\Action::make('view_card')
                        ->label('View')
                        ->icon('heroicon-o-eye')
                        ->url(function ($record) {
                            if (!$record) return '#';

                            $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;
                            return $baseUrl . $record->user->username . '/page/' . $record->slug;
                        })
                        ->openUrlInNewTab()
                        ->color('info'),
                    Tables\Actions\EditAction::make(),

                    Tables\Actions\DeleteAction::make(),
                ])
                    ->label('Actions') // optional — can be hidden if you prefer
                    ->icon('heroicon-o-cog-6-tooth') // optional
                    ->color('gray'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
