<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use App\Models\Image;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\ImageColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Closure;
use Filament\Support\Enums\IconPosition;

class ImagesRelationManager extends RelationManager
{
    protected static string $relationship = 'images';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return in_array($ownerRecord->user_type, ['Sponsored', 'Gold', 'Platinum']);
    }

    protected static ?string $title = 'Gallery';
    protected static ?string $modelLabel = 'Images';

    protected static $typeConfig = [
        'Squared' => [
            'label' => 'Upload Squared Image(s)',
            'placeholder' => 'Upload Squared Image(s)',
            'aspect_ratio' => '1:1',
            'width' => 600,
            'height' => 600,
        ],
        'Landscape' => [
            'label' => 'Upload Landscape Image(s)',
            'placeholder' => 'Upload Landscape Image(s)',
            'aspect_ratio' => '1080:600',
            'width' => 1080,
            'height' => 600,
        ],
    ];

    public function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Select::make('type')
                ->required()
                ->columnSpanFull()
                ->options([
                    'Squared' => 'Squared',
                    'Landscape' => 'Landscape',
                ])
                ->default(fn($record) => $record?->type)
                ->reactive(),

            FileUpload::make('image_path')
                ->reactive()
                ->visible(fn($get) => $get('type') === 'Squared')
                ->image()
                ->imageEditor()
                ->columnSpanFull()
                ->directory('images')
                ->label('Upload Squared Image')
                ->placeholder('Upload Squared Image')
                ->imageCropAspectRatio('1:1')
                ->imageResizeTargetWidth(600)
                ->imageResizeTargetHeight(600),

            FileUpload::make('image_path')
                ->reactive()
                ->visible(fn($get) => $get('type') === 'Landscape')
                ->image()
                ->imageEditor()
                ->columnSpanFull()
                ->directory('images')
                ->label('Upload Landscape Image')
                ->placeholder('Upload Landscape Image')
                ->imageCropAspectRatio('1080:600')
                ->imageResizeTargetWidth(1080)
                ->imageResizeTargetHeight(600),

            Forms\Components\TextInput::make('link')
                ->columnSpanFull(),

            Forms\Components\Toggle::make('is_active')
                ->columnSpanFull()
                ->label('Active')
                ->default(true),

        ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->whereNull('page_id')->orderBy('order', 'asc')
            )
            ->description("Maximum of 10 images. Images posted here will be displayed on the owner's card.")
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                ImageColumn::make('image_path')
                    ->label('Image'),

                Tables\Columns\TextColumn::make('type')
                    ->label('Type')
                    ->getStateUsing(fn($record) => trim($record->type) !== '' ? $record->type : 'Squared'),

                Tables\Columns\TextColumn::make('link')
                    ->searchable()
                    ->limit(40)
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->link)
                    ->openUrlInNewTab(),

            ])
            ->filters([])
            ->headerActions([

                Action::make('upload-images')
                    ->label('Upload Images')
                    ->modalHeading('Upload Images')
                    ->modalButton('Upload')
                    // ->modalWidth('2xl')
                    ->form([
                        Forms\Components\Select::make('type')
                            ->columnSpanFull()
                            ->required()
                            ->options([
                                'Squared' => 'Squared',
                                'Landscape' => 'Landscape',
                            ])
                            ->reactive(),

                        FileUpload::make('image_path')
                            ->reactive()
                            ->multiple()
                            ->visible(fn(callable $get) => $get('type') !== null)
                            ->label(fn(callable $get) => self::$typeConfig[$get('type')]['label'] ?? 'Upload Image(s)')
                            ->placeholder(function (callable $get, RelationManager $livewire) {
                                $currentCount = $livewire->ownerRecord->images()->whereNull('page_id')->count();
                                $remaining = max(0, 10 - $currentCount);
                                return "Upload {$remaining} Image(s)";
                            })
                            ->directory('images')
                            ->uploadingMessage('Uploading. Please wait...')
                            ->image()
                            ->imageEditor()
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio(fn(callable $get) => self::$typeConfig[$get('type')]['aspect_ratio'] ?? '1:1')
                            ->imageResizeTargetWidth(fn(callable $get) => self::$typeConfig[$get('type')]['width'] ?? 600)
                            ->imageResizeTargetHeight(fn(callable $get) => self::$typeConfig[$get('type')]['height'] ?? 600)
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                Str::random(40) . '.' . $file->getClientOriginalExtension()
                            )
                            ->imageEditorAspectRatios(fn(callable $get) => [self::$typeConfig[$get('type')]['aspect_ratio'] ?? '1:1']),
                    ])
                    ->action(function (Action $action, array $data, RelationManager $livewire) {
                        $images = is_array($data['image_path']) ? $data['image_path'] : [$data['image_path']];
                        foreach ($images as $image) {
                            $livewire->ownerRecord->images()->create([
                                'image_path' => $image,
                                'type' => $data['type'],
                                'user_id' => $livewire->ownerRecord->user_id,
                            ]);
                        }

                        Notification::make()
                            ->success()
                            ->title('Image(s) uploaded successfully.')
                            ->send();
                    })
                    // ✅ disable modal "Upload" button while uploading
                    ->extraAttributes([
                        'wire:loading.attr' => 'disabled',
                        'wire:target' => 'data.image_path',
                    ])
                    ->disabled(
                        fn(RelationManager $livewire) =>
                        $livewire->ownerRecord->images()->whereNull('page_id')->count() >= 10
                    ),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->modalHeading('Are you sure you want to delete this image?')
                    ->modalDescription('This action cannot be undone.')
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
