<?php

namespace App\Filament\Resources\UpgradeRequestResource\Pages;

use App\Filament\Resources\UpgradeRequestResource;
use Filament\Actions;
use Filament\Resources\Pages\ViewRecord;
use App\Models\SubCard;
use App\Models\User;
use App\Models\UpgradeRequest;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Storage;
use App\Settings\CardDefaultsSetting;
use App\Notifications\UserNotification;

class ViewUpgradeRequest extends ViewRecord
{
    protected static string $resource = UpgradeRequestResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('upgrade_card')
                ->label('Upgrade Card')
                ->icon('heroicon-o-credit-card')
                ->color('success')
                ->requiresConfirmation()
                ->visible(fn($record) => ! $record->is_upgraded)
                ->action(function ($record) {
                    $settings = app(CardDefaultsSetting::class);
                    // Map plans to card types
                    $planMap = [
                        'basic'       => 'Basic',
                        'basic_m'     => 'Basic',
                        'gold'        => 'Gold',
                        'gold_m'      => 'Gold',
                        'platinum'    => 'Platinum',
                        'platinum_m'  => 'Platinum',
                    ];

                    $planText = [
                        'basic'       => $settings->term_basic,
                        'basic_m'     => $settings->term_basic,
                        'gold'        => $settings->term_gold,
                        'gold_m'      => $settings->term_gold,
                        'platinum'    => $settings->term_platinum,
                        'platinum_m'  => $settings->term_platinum,
                    ];

                    if (! isset($planMap[$record->plan])) {
                        Notification::make()
                            ->title('Invalid plan')
                            ->danger()
                            ->send();
                        return;
                    }

                    // Get user
                    $user = User::find($record->user_id);
                    $update = UpgradeRequest::find($record->id);

                    if (! $user) {
                        Notification::make()
                            ->title('User not found')
                            ->danger()
                            ->send();

                        return;
                    }

                    // Upgrade user
                    $user->update([
                        'user_type' => $planMap[$record->plan],
                    ]);

                    $update->update([
                        'is_upgraded' => true,
                    ]);

                    $newUserType = $planText[$record->plan];

                    Notification::make()
                        ->title('Card successfully upgraded')
                        ->body("User {$user->firstname} {$user->lastname} was upgraded to {$newUserType}.")
                        ->success()
                        ->send();


                    // Notify the user using your existing UserNotification
                    if ($user && ! $user->is_admin) {
                        $user->notify(
                            new UserNotification(
                                title: 'Card Upgraded',
                                message: "<span class='font-semibold'>{$user->firstname} {$user->lastname}</span>, your card has been upgraded to <span class='font-semibold'>{$newUserType}</span>.",
                                url: route('dashboard.subscription')
                            )
                        );
                    }
                }),
        ];
    }
}
