<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SubscriptionResource\Pages;
use App\Filament\Resources\SubscriptionResource\RelationManagers;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Laravel\Cashier\Subscription;
use App\Settings\CardDefaultsSetting;
use Filament\Tables\Filters\SelectFilter;

class SubscriptionResource extends Resource
{
    protected static ?string $model = Subscription::class;

    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';

    protected static ?string $navigationGroup = 'Client Management';
    protected static ?int $navigationSort = 5;

    // public static function form(Form $form): Form
    // {
    //     return $form
    //         ->schema([
    //             // Usually subscriptions are read-only unless you want to allow cancellation/updating
    //             Forms\Components\TextInput::make('name')->disabled(),
    //             Forms\Components\TextInput::make('stripe_status')->disabled(),
    //             Forms\Components\TextInput::make('stripe_price')->disabled(),
    //             Forms\Components\DateTimePicker::make('created_at')->disabled(),
    //         ]);
    // }

    public static function table(Table $table): Table
    {
        return $table
            ->description('This table will show the clients who register and made transaction in the platform.')
            ->columns([
                Tables\Columns\TextColumn::make('stripe_status')
                    ->label('Status')
                    ->badge()
                    ->formatStateUsing(function ($state) {
                        // Normalize the case first
                        $formatted = $state === 'past_due' ? 'Past Due' : ucfirst($state);
                        return $formatted;
                    })
                    ->color(fn(string $state): string => match ($state) {
                        'active' => 'success',
                        'canceled' => 'danger',
                        'trialing' => 'info',
                        'past_due' => 'warning',
                        'incomplete' => 'indigo',
                    }),

                Tables\Columns\TextColumn::make('user.user_type')
                    // ->sortable()
                    ->label('Card Type')
                    ->badge()
                    // ->getStateUsing(fn($record) => $record->user_type ?? 'setup_incomplete')
                    ->formatStateUsing(function ($state) {
                        return $state === 'setup_incomplete' ? 'Setup Incomplete' : $state;
                    })
                    ->formatStateUsing(fn($state) => ucfirst($state))
                    ->formatStateUsing(function ($state) {
                        // return $state === 'setup_incomplete' ? 'Setup Incomplete' : $state;

                        $settings = app(CardDefaultsSetting::class);

                        return match ($state) {
                            'Basic' => $settings->term_basic,
                            'Gold' => $settings->term_gold,
                            'Platinum' => $settings->term_platinum,
                            'setup_incomplete' => 'Setup Incomplete',
                            default => $state,
                        };
                    })
                    ->color(function ($state) {
                        return match ($state) {
                            'Free' => 'info',
                            'Basic' => 'success',
                            'Gold' => 'warning',
                            'Platinum' => 'gray',
                            'Sponsored' => 'danger',
                            'Influencer' => 'influencer',
                            'setup_incomplete' => 'indigo',
                        };
                    }),
                Tables\Columns\TextColumn::make('user.firstname')->label('Client')
                    // ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->user->username)
                    ->url(fn($record) => $record->user
                        ? app(CardDefaultsSetting::class)->site_url . $record->user->username
                        : null)
                    ->getStateUsing(function ($record) {
                        return $record->user ? $record->user->firstname . ' ' . $record->user->lastname : 'N/A';
                    })
                    ->searchable(['firstname', 'lastname']),
                // Tables\Columns\TextColumn::make('user.user_type')
                //     ->badge()
                //     ->label('Card Type')
                //     ->color(fn(string $state): string => match ($state) {
                //         'Free' => 'info',
                //         'Basic' => 'success',
                //         'Gold' => 'warning',
                //         'Platinum' => 'gray',
                //         'Sponsored' => 'danger',
                //     })
                //

                // Tables\Columns\TextColumn::make('name')->label('Subscription Name'),

                Tables\Columns\TextColumn::make('created_at')
                    ->sortable()
                    ->label('Start')
                    ->formatStateUsing(fn($state) => $state->format('M d, Y')),
                // ->formatStateUsing(fn($state) => $state->format('M d, Y | h:i a')),
                Tables\Columns\TextColumn::make('subscription_ends')
                    ->sortable()
                    ->label('Ends')
                    ->getStateUsing(function ($record) {
                        if ($record->ends_at) {
                            return $record->ends_at->format('M d, Y');
                            // return $record->ends_at->format('M d, Y | h:i a');
                        } elseif ($record->stripe_status === 'active') {
                            $stripeSubscription = $record->asStripeSubscription();
                            // return \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_end)->format('M d, Y | h:i a');
                            return \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_end)->format('M d, Y');
                        } else {
                            return 'N/A';
                        }
                    }),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                SelectFilter::make('stripe_status')
                    ->label('Status')
                    ->options([
                        'active' => 'Active',
                        'canceled' => 'Canceled',
                    ]),
            ])
            ->actions([
                Tables\Actions\Action::make('edit_user')
                    ->label('Edit')
                    ->icon('heroicon-o-pencil')
                    ->url(fn($record) => \App\Filament\Resources\UserResource::getUrl('edit', ['record' => $record->user_id]))
                    ->color('primary'),

                Tables\Actions\Action::make('view_card')
                    ->label('View')
                    ->icon('heroicon-o-eye')
                    ->url(fn($record) => $record->user ? app(CardDefaultsSetting::class)->site_url . $record->user->username : null)
                    ->openUrlInNewTab()
                    ->color('info')
                    ->disabled(fn($record) => !$record->user),

            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSubscriptions::route('/'),
            'create' => Pages\CreateSubscription::route('/create'),
            // 'edit' => Pages\EditSubscription::route('/{record}/edit'),
        ];
    }
}
