<?php

namespace App\Filament\Resources\SubCardsResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Database\Eloquent\Model;
use Filament\Support\Enums\IconPosition;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Filament\Forms\Components\FileUpload;
use Filament\Tables\Actions\Action;
use Filament\Tables\Filters\Filter;

class ServicesRelationManager extends RelationManager
{
    protected static string $relationship = 'services';

    // public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    // {
    //     return in_array($ownerRecord->user_type, ['Gold', 'Platinum']);
    // }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                FileUpload::make('service_image')
                    ->label('Upload Service Image')
                    ->placeholder('Upload Service Image')
                    ->directory('services')
                    ->uploadingMessage('Uploading Service Image. Please wait...')
                    ->image()
                    ->imageEditor()
                    ->columnSpanFull()
                    ->imageResizeMode('cover')
                    ->imageCropAspectRatio('1:1')
                    ->imageResizeTargetWidth('520')
                    ->imageResizeTargetHeight('520')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    )
                    ->imageEditorAspectRatios([
                        '1:1',
                    ]),

                Forms\Components\TextInput::make('title')
                    ->label('Service Title')
                    ->required()
                    ->maxLength(255)
                    ->columnSpanFull(),
                Forms\Components\TextInput::make('price')
                    ->required()
                    ->numeric()
                    ->prefix('$'),
                Forms\Components\TextInput::make('payment_link')
                    ->required()
                    ->url()
                    ->maxLength(255),

                Forms\Components\RichEditor::make('description')
                    ->required()
                    ->maxLength(1000)
                    ->columnSpanFull()
                    ->disableGrammarly()
                    ->toolbarButtons([
                        'blockquote',
                        'bold',
                        'bulletList',
                        'h1',
                        'h2',
                        'h3',
                        'italic',
                        'orderedList',
                        'redo',
                        'underline',
                        'undo',
                    ])
                    ->helperText('Write a few sentences about the service. Max character 1000.'),

                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All items listed/added here will be displayed under the "Services" button on the card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )

            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title')
                    ->label('Service Title')
                    ->formatStateUsing(fn($state) => ucfirst($state))
                    ->searchable(),

                Tables\Columns\TextColumn::make('price')
                    ->money('USD')
                    ->sortable()
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
