<?php

namespace App\Filament\Resources\SubCardsResource\Pages;

use App\Filament\Resources\SubCardsResource;
use Filament\Resources\Pages\ViewRecord;
use Filament\Actions;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ViewSubCards extends ViewRecord
{
    protected static string $resource = SubCardsResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\EditAction::make(),
            Actions\Action::make('customAction')
                ->icon('heroicon-o-document-duplicate')
                ->label('Duplicate Card')
                ->color('warning')
                ->action(function (Model $record) {
                    DB::transaction(function () use ($record) {

                        // Helper to duplicate a file into a specific folder
                        // $duplicateFile = function (?string $filePath) {
                        //     if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                        //         return null;
                        //     }

                        //     $directory = dirname($filePath); // e.g. profile_pic
                        //     $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                        //     $originalName = pathinfo($filePath, PATHINFO_FILENAME);
                        //     $newFileName = $originalName . '_' . Str::random(6) . '.' . $extension;
                        //     $newPath = $directory . '/' . $newFileName;

                        //     Storage::disk('public')->makeDirectory($directory);
                        //     Storage::disk('public')->copy($filePath, $newPath);

                        //     return $newPath;
                        // };

                        $duplicateFile = function (?string $filePath) {
                            if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                                return null;
                            }

                            $directory = dirname($filePath); // e.g. profile_pic
                            $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                            $originalName = pathinfo($filePath, PATHINFO_FILENAME);

                            // Limit the original name length to avoid filesystem errors
                            $maxBaseLength = 50; // adjust as needed (e.g. 50 chars)
                            $shortName = Str::limit($originalName, $maxBaseLength, '');

                            $newFileName = $shortName . '_' . Str::random(6) . '.' . $extension;
                            $newPath = $directory . '/' . $newFileName;

                            Storage::disk('public')->makeDirectory($directory);
                            Storage::disk('public')->copy($filePath, $newPath);

                            return $newPath;
                        };

                        // Duplicate logic here
                        $newRecord = $record->replicate(); // Clone the record

                        // Modify unique fields to avoid conflicts
                        $newRecord->username = $record->username . '_' . uniqid(); // Add random suffix
                        $newRecord->email = uniqid() . '_' . $record->email; // Or generate a dummy email if needed

                        // Duplicate each image to its respective folder
                        $newRecord->thumbnail = $duplicateFile($record->thumbnail);
                        $newRecord->profile_pic = $duplicateFile($record->profile_pic);
                        $newRecord->cover_pic = $duplicateFile($record->cover_pic);

                        //dont replicate the following
                        $newRecord->phone = NULL;
                        $newRecord->email = NULL;
                        $newRecord->created_at = Carbon::now();
                        $newRecord->updated_at = Carbon::now();
                        $newRecord->firstname = NULL;
                        $newRecord->lastname = NULL;
                        $newRecord->job_title = NULL;
                        $newRecord->bio = NULL;
                        $newRecord->bio_long = NULL;
                        $newRecord->company = NULL;
                        $newRecord->company_website = NULL;

                        $newRecord->is_duplicated = true;

                        $newRecord->save();

                        //replicate settings
                        if ($record->settings) {
                            foreach ($record->settings as $settings) {
                                $newSettings = $settings->replicate();
                                $newSettings->page_id = $newRecord->id;
                                $newSettings->save();
                            }
                        }

                        //replicate buttons
                        // if ($record->buttons) {
                        //     foreach ($record->buttons()->orderBy('order')->get() as $buttons) {
                        //         $newButtons = $buttons->replicate();
                        //         $newButtons->page_id = $newRecord->id;
                        //         $newButtons->order = $buttons->order;
                        //         $newButtons->save();
                        //     }
                        // }

                        if ($record->buttons()->exists()) {
                            foreach ($record->buttons()->orderBy('order')->get() as $button) {
                                // Replicate the button
                                $newButton = $button->replicate();
                                $newButton->page_id = $newRecord->id;
                                $newButton->order = $button->order;
                                $newButton->save();

                                // If it's a "team" type button, replicate its members
                                if ($newButton->type === 'team') {
                                    foreach ($button->members as $member) {
                                        $newMember = $member->replicate();
                                        $newMember->team_id = $newButton->id;
                                        $newMember->member_id = $member->member_id;
                                        $newMember->card_type = $member->card_type;
                                        $newMember->order = $member->order;
                                        $newMember->custom_card_link = $member->custom_card_link;
                                        $newMember->save();
                                    }
                                }
                            }
                        }

                        //replicate socials
                        if ($record->socials) {
                            foreach ($record->socials()->orderBy('order')->get() as $socials) {
                                $newSocials = $socials->replicate();
                                $newSocials->page_id = $newRecord->id;
                                $newSocials->order = $socials->order;
                                $newSocials->save();
                            }
                        }

                        //replicate videos
                        if ($record->videos) {
                            foreach ($record->videos()->orderBy('order')->get() as $videos) {
                                $newVideos = $videos->replicate();
                                $newVideos->page_id = $newRecord->id;
                                $newVideos->order = $videos->order;
                                $newVideos->save();
                            }
                        }

                        //replicate ads
                        if ($record->ads) {
                            foreach ($record->ads()->orderBy('order')->get() as $ads) {
                                $newAds = $ads->replicate();
                                $newAds->image = $duplicateFile($ads->image);
                                $newAds->page_id = $newRecord->id;
                                $newAds->order = $ads->order;
                                $newAds->save();
                            }
                        }

                        //replicate driving directions
                        if ($record->directions) {
                            foreach ($record->directions()->orderBy('order')->get() as $directions) {
                                $newDirections = $directions->replicate();
                                $newDirections->page_id = $newRecord->id;
                                $newDirections->order = $directions->order;
                                $newDirections->save();
                            }
                        }

                        // replicate products and their media
                        if ($record->products) {
                            foreach ($record->products()->orderBy('order')->get() as $product) {
                                $newProduct = $product->replicate();
                                $newProduct->featured_image = $duplicateFile($product->featured_image);
                                $newProduct->page_id = $newRecord->id;
                                $newProduct->order = $product->order;
                                $newProduct->save();

                                // replicate associated product media
                                if ($product->media && $product->media()->exists()) {
                                    foreach ($product->media()->orderBy('order')->get() as $media) {
                                        $newMedia = $media->replicate();
                                        // ✅ Only duplicate the file if it's an image
                                        if ($media->type === 'image') {
                                            $newPath = $duplicateFile($media->path);

                                            // skip if file missing or invalid
                                            if (!$newPath) {
                                                continue;
                                            }

                                            $newMedia->path = $newPath;
                                        } else {
                                            // non-image types (video, pdf, etc.) keep the same path
                                            $newMedia->path = $media->path;
                                        }

                                        $newMedia->product_id = $newProduct->id; // link to new product
                                        $newMedia->order = $media->order;
                                        $newMedia->save();
                                    }
                                }
                            }
                        }

                        //replicate services
                        if ($record->services) {
                            foreach ($record->services()->orderBy('order')->get() as $services) {
                                $newServices = $services->replicate();
                                $newServices->service_image = $duplicateFile($services->service_image);
                                $newServices->page_id = $newRecord->id;
                                $newServices->order = $services->order;
                                $newServices->save();
                            }
                        }

                        //replicate images
                        if ($record->images) {
                            foreach ($record->images()->orderBy('order')->get() as $images) {
                                $newImages = $images->replicate();
                                $newImages->image_path = $duplicateFile($images->image_path);
                                $newImages->page_id = $newRecord->id;
                                $newImages->order = $images->order;
                                $newImages->save();
                            }
                        }

                        //replicate events
                        if ($record->events) {
                            // foreach ($record->services as $services) {
                            foreach ($record->events()->orderBy('order')->get() as $events) {
                                $newEvents = $events->replicate();
                                $newEvents->event_poster = $duplicateFile($events->event_poster);
                                $newEvents->page_id = $newRecord->id;
                                $newEvents->order = $events->order; // ✅ preserve original order
                                $newEvents->save();
                            }
                        }

                        // Optional: You could redirect or notify
                        Notification::make()
                            ->title('Success')
                            ->body('Sub Card duplicated successfully.')
                            ->success()
                            ->send();
                    });
                }),
        ];
    }
}
