<?php

namespace App\Filament\Resources\SubCardsResource\Pages;

use App\Filament\Resources\SubCardsResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class EditSubCards extends EditRecord
{
    protected static string $resource = SubCardsResource::class;

    public $defaults = [
        'button_style'                          => 'rounded',
        'color_settings_custom_button'          => '#facc15',
        'color_settings_custom_button_text'     => '#000000',
        'color_settings_link'                   => '#ca8a04',
        'color_settings_profile_pic_border'     => '#ca8a04',
        'color_settings_card_light'             => '#ffffff',
        'color_settings_card_dark'              => '#18181b',
        'color_settings_text_light'             => '#1e293b',
        'color_settings_text_dark'              => '#cbd5e1',
        'profile_layout'                        => 'default',
        'searchable'                            => true,
        'display_username'                      => true,
        'display_short_bio'                     => true,
        'display_country'                       => true,
        'connect_main'                          => true,
        'connect_bio'                           => false,
        'gallery_main'                          => true,
        'gallery_bio'                           => false,
        'product_price_enabled'                 => true,
        'services_price_enabled'                => true,
        'footer_enabled'                        => true,
        'getacard_enabled'                      => false,
        'shop_text'                             => 'Shop',
        'shop_more'                             => NULL,
        'shop_thumbnail'                        => NULL,
        'services_text'                         => 'Services',
        'contact_text'                          => 'Contact me',
        'gallery_two_slide'                     => true,
        'follow_text'                           => 'Follow me',
        'event_text'                            => 'Events & Promotions',
        'payment_text'                          => 'Payment Links',
    ];

    protected function mutateFormDataBeforeFill(array $data): array
    {
        $subcard = $this->record;

        foreach ($subcard->settings()->get() as $setting) {
            $data[$setting->setting_key] = $setting->setting_value;
        }

        foreach ($this->defaults as $key => $value) {
            if (!array_key_exists($key, $data) || is_null($data[$key])) {
                $data[$key] = $value;
            }
        }

        return $data;
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {

        $subcard = $this->record;

        // If a preset was selected (and not "Custom"), load the preset and override the colors
        if (!empty($data['preset']) && $data['preset'] !== 'Custom') {
            $preset = \App\Models\ColorPreset::find($data['preset']);

            if ($preset) {
                $data['color_settings_custom_button'] = $preset->custom_button;
                $data['color_settings_custom_button_text'] = $preset->custom_button_text;
                $data['color_settings_link'] = $preset->link;
                $data['color_settings_profile_pic_border'] = $preset->profile_pic_border;
                $data['color_settings_card_light'] = $preset->card_light;
                $data['color_settings_card_dark'] = $preset->card_dark;
                $data['color_settings_text_light'] = $preset->text_light;
                $data['color_settings_text_dark'] = $preset->text_dark;
            }
        }

        // Apply defaults
        if ($this->record->wasRecentlyCreated) {
            foreach ($this->defaults as $key => $value) {
                if (!array_key_exists($key, $data) || is_null($data[$key])) {
                    $data[$key] = $value;
                }
            }
        }

        foreach ($this->defaults as $key => $value) {
            if (!array_key_exists($key, $data) || is_null($data[$key])) {
                $data[$key] = $value;
            }
        }

        // 🔥 Handle image deletion for shop_thumbnail before updating
        if (array_key_exists('shop_thumbnail', $data)) {
            $oldSetting = $subcard->settings()
                ->where('setting_key', 'shop_thumbnail')
                ->first();

            $oldPath = $oldSetting?->setting_value;

            // If user removed the image (clicked X)
            if (blank($data['shop_thumbnail']) && filled($oldPath)) {
                Storage::disk('public')->delete($oldPath);
            }
        }

        // Save only setting_value here; don't set user_id yet
        foreach ($data as $key => $value) {
            if (!Schema::hasColumn('subcards', $key)) {
                $subcard->settings()->updateOrCreate(
                    [
                        'setting_key' => $key,
                        'page_id'     => $subcard->id,
                    ],
                    [
                        'setting_value' => $value,
                        // user_id will be fixed in afterSave()
                    ]
                );
                unset($data[$key]);
            }
        }

        return $data;
    }

    protected function afterSave(): void
    {
        $record = $this->record;
        $newUserId = $record->user_id;

        // Update settings user_id directly here
        $record->settings()->where('user_id', '!=', $newUserId)
            ->update(['user_id' => $newUserId]);

        // Update all other relations if user_id changed
        if ($record->wasChanged('user_id')) {
            $relations = [
                'buttons',
                'socials',
                'videos',
                'ads',
                'products',
                'services',
                'images',
                'contacts',
                'payments',
            ];

            foreach ($relations as $relation) {
                if (!method_exists($record, $relation)) {
                    continue;
                }

                $record->{$relation}()
                    ->where('user_id', '!=', $newUserId)
                    ->update(['user_id' => $newUserId]);
            }
        }
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('customAction')
                ->icon('heroicon-o-eye')
                ->label('View') // Button text
                ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->username)
                ->openUrlInNewTab()
                ->color('info'),

            Actions\Action::make('customAction')
                ->icon('heroicon-o-document-duplicate')
                ->label('Duplicate')
                ->color('warning')
                ->action(function (Model $record) {
                    DB::transaction(function () use ($record) {

                        $duplicateFile = function (?string $filePath) {
                            if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                                return null;
                            }

                            $directory = dirname($filePath); // e.g. profile_pic
                            $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                            $originalName = pathinfo($filePath, PATHINFO_FILENAME);

                            // Limit the original name length to avoid filesystem errors
                            $maxBaseLength = 50; // adjust as needed (e.g. 50 chars)
                            $shortName = Str::limit($originalName, $maxBaseLength, '');

                            $newFileName = $shortName . '_' . Str::random(6) . '.' . $extension;
                            $newPath = $directory . '/' . $newFileName;

                            Storage::disk('public')->makeDirectory($directory);
                            Storage::disk('public')->copy($filePath, $newPath);

                            return $newPath;
                        };

                        // Duplicate logic here
                        $newRecord = $record->replicate(); // Clone the record

                        // Modify unique fields to avoid conflicts
                        $newRecord->username = $record->username . '_' . uniqid(); // Add random suffix
                        $newRecord->email = uniqid() . '_' . $record->email; // Or generate a dummy email if needed

                        // Duplicate each image to its respective folder
                        $newRecord->thumbnail = $duplicateFile($record->thumbnail);
                        $newRecord->profile_pic = $duplicateFile($record->profile_pic);
                        $newRecord->cover_pic = $duplicateFile($record->cover_pic);

                        //dont replicate the following
                        $newRecord->phone = NULL;
                        $newRecord->email = NULL;
                        $newRecord->created_at = Carbon::now();
                        $newRecord->updated_at = Carbon::now();
                        $newRecord->firstname = NULL;
                        $newRecord->lastname = NULL;
                        $newRecord->job_title = NULL;
                        $newRecord->bio = NULL;
                        $newRecord->bio_long = NULL;
                        $newRecord->company = NULL;
                        $newRecord->company_website = NULL;

                        $newRecord->is_duplicated = true;

                        $newRecord->save();

                        //replicate settings
                        if ($record->settings) {
                            foreach ($record->settings as $settings) {
                                $newSettings = $settings->replicate();
                                $newSettings->page_id = $newRecord->id;
                                $newSettings->save();
                            }
                        }

                        if ($record->buttons()->exists()) {
                            foreach ($record->buttons()->orderBy('order')->get() as $button) {
                                // Replicate the button
                                $newButton = $button->replicate();
                                $newButton->page_id = $newRecord->id;
                                $newButton->order = $button->order;
                                $newButton->save();

                                // If it's a "team" type button, replicate its members
                                if ($newButton->type === 'team') {
                                    foreach ($button->members as $member) {
                                        $newMember = $member->replicate();
                                        $newMember->team_id = $newButton->id;
                                        $newMember->member_id = $member->member_id;
                                        $newMember->card_type = $member->card_type;
                                        $newMember->order = $member->order;
                                        $newMember->custom_card_link = $member->custom_card_link;
                                        $newMember->save();
                                    }
                                }
                            }
                        }

                        //replicate socials
                        if ($record->socials) {
                            foreach ($record->socials()->orderBy('order')->get() as $socials) {
                                $newSocials = $socials->replicate();
                                $newSocials->page_id = $newRecord->id;
                                $newSocials->order = $socials->order;
                                $newSocials->save();
                            }
                        }

                        //replicate videos
                        if ($record->videos) {
                            foreach ($record->videos()->orderBy('order')->get() as $videos) {
                                $newVideos = $videos->replicate();
                                $newVideos->page_id = $newRecord->id;
                                $newVideos->order = $videos->order;
                                $newVideos->save();
                            }
                        }

                        //replicate ads
                        if ($record->ads) {
                            foreach ($record->ads()->orderBy('order')->get() as $ads) {
                                $newAds = $ads->replicate();
                                $newAds->image = $duplicateFile($ads->image);
                                $newAds->page_id = $newRecord->id;
                                $newAds->order = $ads->order;
                                $newAds->save();
                            }
                        }

                        //replicate driving directions
                        if ($record->directions) {
                            foreach ($record->directions()->orderBy('order')->get() as $directions) {
                                $newDirections = $directions->replicate();
                                $newDirections->page_id = $newRecord->id;
                                $newDirections->order = $directions->order;
                                $newDirections->save();
                            }
                        }

                        // replicate products and their media
                        if ($record->products) {
                            foreach ($record->products()->orderBy('order')->get() as $product) {
                                $newProduct = $product->replicate();
                                $newProduct->featured_image = $duplicateFile($product->featured_image);
                                $newProduct->page_id = $newRecord->id;
                                $newProduct->order = $product->order;
                                $newProduct->save();

                                // replicate associated product media
                                if ($product->media && $product->media()->exists()) {
                                    foreach ($product->media()->orderBy('order')->get() as $media) {
                                        $newMedia = $media->replicate();
                                        // ✅ Only duplicate the file if it's an image
                                        if ($media->type === 'image') {
                                            $newPath = $duplicateFile($media->path);

                                            // skip if file missing or invalid
                                            if (!$newPath) {
                                                continue;
                                            }

                                            $newMedia->path = $newPath;
                                        } else {
                                            // non-image types (video, pdf, etc.) keep the same path
                                            $newMedia->path = $media->path;
                                        }

                                        $newMedia->product_id = $newProduct->id; // link to new product
                                        $newMedia->order = $media->order;
                                        $newMedia->save();
                                    }
                                }
                            }
                        }

                        //replicate services
                        if ($record->services) {
                            foreach ($record->services()->orderBy('order')->get() as $services) {
                                $newServices = $services->replicate();
                                $newServices->service_image = $duplicateFile($services->service_image);
                                $newServices->page_id = $newRecord->id;
                                $newServices->order = $services->order;
                                $newServices->save();
                            }
                        }

                        //replicate images
                        if ($record->images) {
                            foreach ($record->images()->orderBy('order')->get() as $images) {
                                $newImages = $images->replicate();
                                $newImages->image_path = $duplicateFile($images->image_path);
                                $newImages->page_id = $newRecord->id;
                                $newImages->order = $images->order;
                                $newImages->save();
                            }
                        }

                        //replicate events
                        if ($record->events) {
                            // foreach ($record->services as $services) {
                            foreach ($record->events()->orderBy('order')->get() as $events) {
                                $newEvents = $events->replicate();
                                $newEvents->event_poster = $duplicateFile($events->event_poster);
                                $newEvents->page_id = $newRecord->id;
                                $newEvents->order = $events->order; // ✅ preserve original order
                                $newEvents->save();
                            }
                        }

                        // Optional: You could redirect or notify
                        Notification::make()
                            ->title('Success')
                            ->body('Sub Card duplicated successfully.')
                            ->success()
                            ->send();
                    });
                }),

            Actions\DeleteAction::make()
                ->icon('heroicon-o-trash'),
            Actions\Action::make('save')
                ->label('Save')
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->action('save'),
        ];
    }
}
