<?php

namespace App\Filament\Resources\SponsorResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\IconPosition;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use App\Models\Button;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;

class ContactsRelationManager extends RelationManager
{
    protected static string $relationship = 'buttons';
    protected static ?string $title = 'Contacts';
    protected static ?string $modelLabel = 'Contact';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('type')
                    ->required()
                    ->label('Link Type')
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $get, callable $set) {
                        $titles = [
                            'email' => 'Email',
                            'phone' => 'Phone',
                            'sms' => 'SMS/Text',
                            'whatsapp' => 'WhatsApp',
                            'telegram' => 'Telegram',
                            'zoom' => 'Zoom',
                            'community' => 'Community',
                        ];

                        $set('title', $titles[$state] ?? '');
                    })
                    ->options([
                        'email' => 'Email',
                        'phone' => 'Phone',
                        'sms' => 'SMS/Text',
                        'whatsapp' => 'WhatsApp',
                        'telegram' => 'Telegram',
                        'zoom' => 'Zoom',
                        'community' => 'Community',
                    ]),
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('link')
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->label('Active')
                    ->default(true)
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All links listed/added here will be displayed as default contact points on the sponsored card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('view', 'contacts')->orderBy('order')
            )
            ->columns([
                // USED FOR DEBUGGING
                // Tables\Columns\TextColumn::make('page_id')->label('Page ID'),
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive
                // Tables\Columns\TextColumn::make('title')
                //     ->searchable(),
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    // ->url(fn($record) => $record->link)
                    ->url(
                        fn($record) =>
                        $record->type === 'email' ? 'mailto:' . $record->link : ($record->type === 'sms' ? 'sms:' . $record->link : ($record->type === 'phone' ? 'tel:' . $record->link : $record->link))
                    )
                    ->openUrlInNewTab(),

                Tables\Columns\TextColumn::make('type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->disabled(function (RelationManager $livewire) {
                        return $livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->where('view', 'contacts')->count() >= 4;
                    })
                    ->before(function (CreateAction $action, $livewire) {
                        // Access form data from the action
                        $data = $action->getFormData();

                        // Validate and process the link
                        try {
                            // Assuming $data is the input data for button creation
                            Button::handleLinkFormatting($data);

                            // Proceed to create the button if no issues
                            $livewire->ownerRecord->buttons()->create($data);

                            Notification::make()
                                ->title('Success')
                                ->body('Contact link created successfully.')
                                ->success()  // Success notification style
                                ->send();

                            // Cancel the action after manual creation to prevent default action execution
                            $action->cancel();

                            // Check if the user is on a free plan and has exceeded the button limit
                            // if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->count() >= 4) {
                            //     Notification::make()
                            //         ->warning()
                            //         ->title('Free card holders can only add up to 4 button links!')
                            //         ->persistent()
                            //         ->send();

                            //     // Cancel the action to prevent further processing
                            //     $action->cancel(); // This ensures that the action is canceled and no further action is executed
                            // } else {
                            //     // Proceed to create the button if no issues
                            //     $livewire->ownerRecord->buttons()->create($data);

                            //     Notification::make()
                            //         ->title('Success')
                            //         ->body('Contact link created successfully.')
                            //         ->success()  // Success notification style
                            //         ->send();

                            //     // Cancel the action after manual creation to prevent default action execution
                            //     $action->cancel();  // Prevent the default action (which would also try to create the button)
                            // }

                        } catch (\InvalidArgumentException $e) {
                            Notification::make()
                                ->title('Validation Error')
                                ->body($e->getMessage())  // Show the exception message
                                ->danger()  // Set notification style to error
                                ->send();

                            // Cancel the action to prevent the model creation
                            $action->halt(); // Halt the action to prevent further processing
                        }
                    })

                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['view'] = 'contacts';
                        return $data;
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()->before(function (EditAction $action, $livewire) {
                    $data = $action->getFormData();
                    try {
                        Button::handleLinkFormatting($data);

                        // Fetch the existing button to update
                        $button = $action->getRecord();

                        $button->update($data);

                        Notification::make()
                            ->title('Success')
                            ->body('Contact link has been updated successfully.')
                            ->success()
                            ->send();

                        $action->cancel();

                        // if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->count() >= 4) {
                        //     Notification::make()
                        //         ->warning()
                        //         ->title('Free card holders can only add up to 4 button links!')
                        //         ->persistent()
                        //         ->send();

                        //     $action->cancel();
                        // } else {
                        //     $button->update($data);

                        //     Notification::make()
                        //         ->title('Success')
                        //         ->body('Contact link has been updated successfully.')
                        //         ->success()
                        //         ->send();

                        //     $action->cancel();
                        // }

                    } catch (\InvalidArgumentException $e) {
                        Notification::make()
                            ->title('Validation Error')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();

                        $action->halt();
                    }
                }),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
