<?php

namespace App\Filament\Resources\SponsorResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\IconPosition;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use App\Models\Button;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;

class ButtonsRelationManager extends RelationManager
{
    protected static string $relationship = 'buttons';

    protected static ?string $title = 'Links';
    protected static ?string $modelLabel = 'Link';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('type')
                    ->required()
                    ->dehydrated(true)
                    ->extraAttributes(fn($record) => $record && in_array($record->type, ['vcf', 'help', 'exchange', 'bio', 'shop', 'services', 'feedback', 'events']) ? ['disabled' => 'disabled', 'style' => 'pointer-events: none; opacity: 0.5;'] : [])
                    ->label('Link Type')
                    ->options(fn($record) => $record && in_array($record->type, ['vcf', 'help', 'exchange', 'bio', 'shop', 'services', 'feedback', 'events']) ? [
                        'vcf' => 'VCF',
                        'help' => 'Help',
                        'exchange' => 'Exchange Contacts',
                        'bio' => 'My Bio',
                        'shop' => 'Shop',
                        'services' => 'Services',
                        'feedback' => 'Feedback',
                        'events' => 'Events',
                    ] : [
                        'link' => 'Link',
                        'email' => 'Email',
                        'phone' => 'Phone',
                        'sms' => 'SMS',
                        'whatsapp' => 'WhatsApp',
                        'telegram' => 'Telegram',
                    ]),

                Forms\Components\TextInput::make('title')
                    ->required()
                    // ->extraAttributes(fn($record) => $record && in_array($record->type, ['vcf', 'help', 'exchange', 'bio', 'shop', 'services', 'feedback', 'events']) ? ['disabled' => 'disabled', 'style' => 'pointer-events: none; opacity: 0.5;'] : [])
                    ->maxLength(255),
                Forms\Components\TextInput::make('link')
                    ->columnSpanFull()
                    ->extraAttributes(fn($record) => $record && in_array($record->type, ['vcf', 'help', 'exchange', 'bio', 'shop', 'services', 'feedback', 'events']) ? ['disabled' => 'disabled', 'style' => 'pointer-events: none; opacity: 0.5;'] : [])
                    ->required(fn($record) => !$record || !in_array($record->type, ['vcf', 'help', 'exchange', 'bio', 'shop', 'services', 'feedback', 'events'])),
                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->label('Active')
                    ->default(true)
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All links listed/added here will be displayed as default button on the sponsored card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('view', 'links')->orderBy('order')
            )
            ->recordTitleAttribute('title')
            ->columns([
                // USED FOR DEBUGGING
                // Tables\Columns\TextColumn::make('sponsor_id')->label('Sponsor ID'),
                // Tables\Columns\IconColumn::make('is_active')
                //     ->label('Active')
                //     ->boolean(),

                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    // ->url(fn($record) => $record->link)
                    ->url(
                        fn($record) =>
                        $record->type === 'email' ? 'mailto:' . $record->link : ($record->type === 'sms' ? 'sms:' . $record->link : ($record->type === 'phone' ? 'tel:' . $record->link : $record->link))
                    )
                    ->openUrlInNewTab(),

                Tables\Columns\TextColumn::make('type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
                SelectFilter::make('type')
                    ->label('Link Type')
                    ->options([
                        'link' => 'Link',
                        'email' => 'Email',
                        'phone' => 'Phone',
                        'sms' => 'SMS',
                        'whatsapp' => 'WhatsApp',
                        'telegram' => 'Telegram',
                    ]),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->before(function (CreateAction $action, $livewire) {
                        // Access form data from the action
                        $data = $action->getFormData();

                        // Validate and process the link
                        try {
                            // Assuming $data is the input data for button creation
                            Button::handleLinkFormatting($data);

                            // Check if the user is on a free plan and has exceeded the button limit
                            if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->count() >= 4) {
                                Notification::make()
                                    ->warning()
                                    ->title('Free card holders can only add up to 4 button links!')
                                    ->persistent()
                                    ->send();

                                // Cancel the action to prevent further processing
                                $action->cancel(); // This ensures that the action is canceled and no further action is executed
                            } else {
                                // Proceed to create the button if no issues
                                $livewire->ownerRecord->buttons()->create($data);

                                Notification::make()
                                    ->title('Success')
                                    ->body('Button created successfully.')
                                    ->success()  // Success notification style
                                    ->send();

                                // Cancel the action after manual creation to prevent default action execution
                                $action->cancel();  // Prevent the default action (which would also try to create the button)
                            }
                        } catch (\InvalidArgumentException $e) {
                            Notification::make()
                                ->title('Validation Error')
                                ->body($e->getMessage())  // Show the exception message
                                ->danger()  // Set notification style to error
                                ->send();

                            // Cancel the action to prevent the model creation
                            $action->halt(); // Halt the action to prevent further processing
                        }
                    })

                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['display_as'] = 'button';
                        return $data;
                    }),

            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->before(function (EditAction $action, $livewire) {
                        $data = $action->getFormData();

                        try {
                            Button::handleLinkFormatting($data);

                            // Fetch the existing button to update
                            $button = $action->getRecord();

                            if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->count() >= 4) {
                                Notification::make()
                                    ->warning()
                                    ->title('Free card holders can only add up to 4 button links!')
                                    ->persistent()
                                    ->send();

                                $action->cancel();
                            } else {
                                $button->update($data);

                                Notification::make()
                                    ->title('Success')
                                    ->body('Button has been updated successfully.')
                                    ->success()
                                    ->send();

                                $action->cancel();
                            }
                        } catch (\InvalidArgumentException $e) {
                            Notification::make()
                                ->title('Validation Error')
                                ->body($e->getMessage())
                                ->danger()
                                ->send();

                            $action->halt();
                        }
                    })
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['display_as'] = 'button';
                        return $data;
                    }),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn($record) => !in_array($record->type, [
                        'vcf',
                        'help',
                        'exchange',
                        'bio',
                        'shop',
                        'services',
                        'feedback',
                        'events'
                    ])),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
