<?php

namespace App\Filament\Resources\SponsorResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Support\Enums\IconPosition;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;
use Filament\Notifications\Notification;
use App\Settings\CardDefaultsSetting;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Filament\Forms\Components\FileUpload;
use Illuminate\Support\Str;

class AdRelationManager extends RelationManager
{
    protected static string $relationship = 'ads';

    public function form(Form $form): Form
    {
        $settings = app(CardDefaultsSetting::class);
        $width = $settings->ad_banner_w;
        $height = $settings->ad_banner_h;
        $aspectRatio = "{$width}:{$height}";

        return $form
            ->schema([
                FileUpload::make('image')
                    ->label("Upload Ad Banner ({$width}x{$height})")
                    ->placeholder('Upload Ad Banner')
                    ->directory('ads')
                    ->disk('public')
                    ->image()
                    ->required()
                    ->imageEditor()
                    ->imageCropAspectRatio($aspectRatio) // This sets the default crop shape
                    ->imageEditorAspectRatios([$aspectRatio]) // Only allow this one crop shape
                    ->imageResizeMode('cover') // Resize after crop
                    ->imageResizeTargetWidth($width)
                    ->imageResizeTargetHeight($height)
                    ->columnSpanFull()
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    ),
                Forms\Components\TextInput::make('link')->required()
                    ->columnSpanFull(),
                Forms\Components\TextInput::make('caption')->label('Ad Title')
                    ->required(),
                Forms\Components\Select::make('position')
                    ->required()
                    ->label('Position')
                    ->options([
                        'top' => 'Top',
                        'bottom' => 'Bottom',
                    ]),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active Ad')
                    ->default(true)
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->description("Ads posted here will be displayed on sponsor's cards.")
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->formatStateUsing(fn($state) => '#' . ($state)),
                Tables\Columns\TextColumn::make('caption')
                    ->label('Ad Title'),
                Tables\Columns\TextColumn::make('display_in')
                    ->formatStateUsing(fn($state) => ucfirst($state))
                    ->badge()
                    ->label('Card')
                    ->color(fn(string $state): string => match ($state) {
                        'Free' => 'info',
                        'Sponsored' => 'danger',
                    }),
                Tables\Columns\TextColumn::make('sponsor.sponsor_name')
                    ->label('Sponsor Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('position')
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
                SelectFilter::make('display_in')
                    ->label('Card')
                    ->options([
                        'Free' => 'Free',
                        'Sponsored' => 'Sponsored',
                    ]),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['sponsor_id'] = $livewire->ownerRecord->user_id;
                        $data['display_in'] = 'Sponsored';
                        $data['is_admin'] = true;
                        return $data;
                    }),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
