<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SponsorResource\Pages;
use App\Filament\Resources\SponsorResource\RelationManagers;
use App\Filament\Resources\SponsorResource\RelationManagers\AdRelationManager;
use App\Filament\Resources\SponsorResource\RelationManagers\ButtonsRelationManager;
use App\Filament\Resources\SponsorResource\RelationManagers\ContactsRelationManager;
use App\Filament\Resources\SponsorResource\RelationManagers\SocialsRelationManager;
use App\Models\Sponsor;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Settings\CardDefaultsSetting;
use Filament\Forms\Components\Select;
use App\Models\User;
use Filament\Notifications\Notification;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Filament\Forms\Set;

class SponsorResource extends Resource
{
    protected static ?string $model = Sponsor::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    protected static ?string $navigationGroup = 'Ads Management';
    protected static ?string $navigationLabel = 'Sponsor Management';
    protected static ?int $navigationSort = 7;

    public static function shouldRegisterNavigation(): bool
    {
        $settings = app(CardDefaultsSetting::class);

        // Hide navigation if sponsored is enabled
        if ($settings->sponsored_enabled) {
            return true;
        }

        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('user_id')
                    ->columnSpanFull()
                    ->label("Sponsor Card")
                    // ->helperText(new HtmlString('Sponsor card will be used as affiliate. Only <strong>Gold</strong> & <strong>Platinum</strong> card holders.'))
                    ->helperText(function () {
                        $settings = app(CardDefaultsSetting::class);

                        return new HtmlString(
                            'Sponsor card will be used as affiliate. Only <strong>' . e($settings->term_gold) . '</strong> & <strong>' . e($settings->term_platinum) . '</strong> card holders.'
                        );
                    })
                    ->placeholder('Search Client')
                    ->getSearchResultsUsing(
                        fn(string $search): array =>
                        User::where(function ($query) use ($search) {

                            if (strpos($search, '@') === 0 && strlen($search) > 1) {
                                $query->where('username', 'like', "%" . substr($search, 1) . "%");
                            } else {
                                $query->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            }
                        })
                            ->where(function ($query) {
                                $query->where('user_type', 'Platinum')
                                    ->orWhere('user_type', 'Gold');
                            })
                            ->limit(50)
                            ->pluck(DB::raw('CONCAT(IFNULL(firstname, ""), " ", IFNULL(lastname, ""), " - @", IFNULL(username, ""))'), 'id')

                            ->toArray()
                    )
                    ->getOptionLabelUsing(function ($value) {
                        $user = User::find($value);
                        return $user ? "{$user->firstname} {$user->lastname}" : '';
                    })
                    ->searchable()
                    ->required()
                    ->afterStateUpdated(function (Set $set, $state) {

                        $user = User::find($state);

                        if ($user->user_type == "Platinum" && $user->subcards()->count() >= 12) {
                            // Trigger a notification
                            Notification::make()
                                ->title('Limit reached')
                                ->body('Platinum card holders can only add up to 12 sub cards!')
                                ->warning()
                                ->send();
                            // Optionally, you can reset the state or show a validation error:
                            $set('user_id', null); // Reset the user_id if limit is reached
                        }
                        if ($user->user_type == "Gold" && $user->subcards()->count() >= 3) {
                            // Trigger a notification
                            Notification::make()
                                ->title('Limit reached')
                                ->body('Gold card holders can only add up to 3 sub cards!')
                                ->warning()
                                ->send();
                            // Optionally, you can reset the state or show a validation error:
                            $set('user_id', null); // Reset the user_id if limit is reached
                        }
                    }),

                Forms\Components\TextInput::make('sponsor_name')
                    ->label('Sponsor Name')
                    ->required(),

                Forms\Components\Select::make('sponsor_type')
                    ->required()
                    ->label('Sponsor Type')
                    ->default('Individual')
                    ->options([
                        'Individual' => 'Individual',
                        'Organization' => 'Organization',
                        'Company' => 'Company',
                    ]),

                Forms\Components\Toggle::make('is_active')
                    ->label('Active Sponsor')
                    ->default(true)
                    ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->description('Sponsors posted here will be used in creating ads and sponsored cards.')
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('sponsor_name')
                    ->label('Sponsor'),
                Tables\Columns\TextColumn::make('sponsor_type')
                    ->label('Sponsor Type'),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getWidgets(): array
    {
        return [
            SponsorResource\Widgets\SponsorAdAnalytics::class,
            SponsorResource\Widgets\SponsorAdAnalyticsOneWeek::class,
        ];
    }

    public static function getRelations(): array
    {
        return [
            ButtonsRelationManager::class,
            ContactsRelationManager::class,
            SocialsRelationManager::class,
            AdRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSponsors::route('/'),
            'create' => Pages\CreateSponsor::route('/create'),
            'view' => Pages\ViewSponsor::route('/{record}'),
            'edit' => Pages\EditSponsor::route('/{record}/edit'),
        ];
    }
}
