<?php

namespace App\Filament\Resources\SaleResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Filament\Tables\Actions\Action;
use AmidEsfahani\FilamentTinyEditor\TinyEditor;
use Filament\Forms\Components\ViewField;
use Illuminate\Support\Str;

class SaleVideosRelationManager extends RelationManager
{
    protected static string $relationship = 'sale_videos';

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return 'Videos';
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([

                Forms\Components\TextInput::make('product_name')
                    ->label('Product Name')
                    ->maxLength(100)
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\TextInput::make('price')
                    ->required()
                    ->numeric()
                    ->prefix('$'),

                Forms\Components\TextInput::make('stock')
                    ->required()
                    ->numeric()
                    ->default(1)
                    ->minValue(1)   // ✅ sets minimum numeric value
                    ->maxValue(999), // ✅ sets maximum numeric value

                Forms\Components\Textarea::make('description')
                    ->label('Product Description')
                    ->maxLength(300)
                    ->columnSpanFull(),

                // TinyEditor::make('description')
                //     ->label('Product Description')
                //     ->profile('custom') // define this profile in config
                //     ->reactive()
                //     ->live()
                //     ->maxLength(300)
                //     ->columnSpan('full')
                //     ->helperText('Product Description. Max character 300.')
                //     ->afterStateUpdated(function ($state, $set) {
                //         $plainText = strip_tags($state ?? '');
                //         $set('description_count', Str::length($plainText)); // keep server-side fallback
                //     })
                //     ->rule(fn($get) => function ($attribute, $value, $fail) {
                //         $plainText = strip_tags($value ?? '');
                //         if (Str::length($plainText) > 300) {
                //             $fail('The product description must not exceed 300 characters.');
                //         }
                //     }),

                // ViewField::make('product_desc_count')
                //     ->view('forms.product-desc-counter')
                //     ->columnSpan('full'),

                Forms\Components\TextInput::make('video_link')
                    ->label('Video Link')
                    ->placeholder('Youtube or Vimeo video link')
                    ->url()
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\TextInput::make('shop_online_button')
                    ->label('Shop Online Button')
                    ->maxLength(100)
                    ->placeholder('Default: Shop Online')
                    ->columnSpanFull()
                    ->required()
                    ->default('Shop Online'),

                Forms\Components\TextInput::make('shop_online_link')
                    ->label('Shop Online Link')
                    ->placeholder('https://linktoyourproduct.com')
                    ->url()
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->orderBy('order', 'asc')
            )
            ->description("Maximum of 10 videos.")
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->searchable(),

                Tables\Columns\TextColumn::make('product_name')
                    ->label('Product Name')
                    ->searchable(),

                Tables\Columns\TextColumn::make('video_link')
                    ->label('Video Link')
                    ->searchable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->label('New Video'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->modalHeading('Are you sure you want to delete this video?')
                    ->modalDescription('This action cannot be undone.'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->modalHeading('Are you sure you want to delete these videos?')
                        ->modalDescription('This action cannot be undone.'),
                ]),
            ]);
    }
}
