<?php

namespace App\Filament\Resources\SaleResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\ImageColumn;
use Filament\Notifications\Notification;
use Intervention\Image\Facades\Image;
use AmidEsfahani\FilamentTinyEditor\TinyEditor;
use Filament\Forms\Components\ViewField;

class SaleImagesRelationManager extends RelationManager
{
    protected static string $relationship = 'sale_images';

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return 'Images';
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                FileUpload::make('image_path')
                    ->required()
                    ->label('Upload Image')
                    ->placeholder('Upload Image')
                    ->directory('sales')
                    ->uploadingMessage('Uploading. Please wait...')
                    ->image()
                    ->imageEditor()
                    ->columnSpanFull()
                    ->imageResizeMode('cover')
                    ->imageCropAspectRatio('1:1')
                    ->imageResizeTargetWidth('600')
                    ->imageResizeTargetHeight('600')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    )
                    ->saveUploadedFileUsing(function (TemporaryUploadedFile $file) {
                        $image = Image::make($file->getRealPath())
                            ->resize(800, null, function ($constraint) {
                                $constraint->aspectRatio();
                                $constraint->upsize();
                            })
                            ->encode('jpg', 75);

                        $filename = Str::random(40) . '.jpg';

                        \Storage::disk('public')->put('sales/' . $filename, (string)$image);

                        return 'sales/' . $filename;
                    })
                    ->imageEditorAspectRatios(['1:1']),

                Forms\Components\TextInput::make('product_name')
                    ->label('Product Name')
                    ->maxLength(100)
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\TextInput::make('price')
                    ->required()
                    ->numeric()
                    ->prefix('$'),

                Forms\Components\TextInput::make('stock')
                    ->required()
                    ->numeric()
                    ->default(1)
                    ->minValue(1)
                    ->maxValue(999),

                Forms\Components\Textarea::make('description')
                    ->label('Product Description')
                    ->maxLength(300)
                    ->columnSpanFull(),

                // TinyEditor::make('description')
                //     ->label('Product Description')
                //     ->profile('custom') // define this profile in config
                //     ->reactive()
                //     ->live()
                //     ->maxLength(300)
                //     ->columnSpan('full')
                //     ->helperText('Product Description. Max character 300.')
                //     ->afterStateUpdated(function ($state, $set) {
                //         $plainText = strip_tags($state ?? '');
                //         $set('description_count', Str::length($plainText));
                //     })
                //     ->rule(fn($get) => function ($attribute, $value, $fail) {
                //         $plainText = strip_tags($value ?? '');
                //         if (Str::length($plainText) > 300) {
                //             $fail('The product description must not exceed 300 characters.');
                //         }
                //     }),

                // TinyEditor::make('description')
                //     ->label('Product Description')
                //     ->profile('custom') // define your TinyEditor profile in config
                //     ->reactive()
                //     ->live()
                //     ->maxLength(300)
                //     ->columnSpan('full')
                //     ->helperText('Product Description. Max character 300.')
                //     ->afterStateUpdated(function ($state, $set) {
                //         $plainText = strip_tags($state ?? '');
                //         // store character count server-side as fallback
                //         $set('description_count', Str::length($plainText));
                //     })
                //     ->rule(fn($get) => function ($attribute, $value, $fail) {
                //         $plainText = strip_tags($value ?? '');
                //         if (Str::length($plainText) > 300) {
                //             $fail('The product description must not exceed 300 characters.');
                //         }
                //     }),

                // ViewField::make('product_desc_count')
                //     ->view('forms.product-desc-counter')
                //     ->columnSpan('full'),

                Forms\Components\TextInput::make('shop_online_button')
                    ->label('Shop Online Button')
                    ->maxLength(100)
                    ->placeholder('Default: Shop Online')
                    ->columnSpanFull()
                    ->required()
                    ->default('Shop Online'),

                Forms\Components\TextInput::make('shop_online_link')
                    ->label('Shop Online Link')
                    ->placeholder('https://linktoyourproduct.com')
                    ->url()
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->orderBy('order', 'asc')
            )
            ->description("Maximum of 10 images per Sale.")
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->searchable(),

                ImageColumn::make('image_path')
                    ->label('Image'),

                Tables\Columns\TextColumn::make('product_name')
                    ->label('Product Name')
                    ->searchable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->label('New Image')
                    ->before(function (Tables\Actions\CreateAction $action, RelationManager $livewire) {

                        $imageCount = $livewire->ownerRecord->sale_images()->count();

                        if ($imageCount >= 10) {
                            $action->halt();

                            Notification::make()
                                ->warning()
                                ->title("You can only add up to 10 images per Sale!")
                                ->persistent()
                                ->send();
                        }
                    })
                    ->disabled(fn(RelationManager $livewire): bool => $livewire->ownerRecord->sale_images()->count() >= 10),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->modalHeading('Are you sure you want to delete this image?')
                    ->modalDescription('This action cannot be undone.'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->modalHeading('Are you sure you want to delete these images?')
                        ->modalDescription('This action cannot be undone.'),
                ]),
            ]);
    }
}
