<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ReferralResource\Pages;
use App\Filament\Resources\ReferralResource\RelationManagers\ReferralsRelationManager;
use App\Filament\Resources\ReferralResource\RelationManagers;
use App\Models\Referral;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components;
use Filament\Infolists\Components\ImageEntry;
use App\Settings\CardDefaultsSetting;

class ReferralResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-plus';

    // protected static ?string $navigationLabel = 'Main Card Referrals';
    protected static ?string $navigationLabel = 'Referrals';

    protected static ?string $modelLabel = 'Referrals';

    // protected static ?string $navigationGroup = 'Referrals Management';
    protected static ?string $navigationGroup = 'Client Management';

    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([

                Tables\Columns\TextColumn::make('referral_count')
                    ->label('Referral Count')
                    ->sortable(),

                Tables\Columns\TextColumn::make('firstname')
                    ->label('Referrer')
                    ->formatStateUsing(function ($state, $record) {
                        return $record->firstname . ' ' . $record->lastname;
                    }),

                Tables\Columns\TextColumn::make('user_type')
                    ->badge()
                    ->label('Card Type')
                    ->formatStateUsing(function ($state) {
                        $settings = app(CardDefaultsSetting::class);

                        return match ($state) {
                            'Basic' => $settings->term_basic,
                            'Gold' => $settings->term_gold,
                            'Platinum' => $settings->term_platinum,
                            'setup_incomplete' => 'Setup Incomplete',
                            default => $state,
                        };
                    })
                    ->color(fn(string $state): string => match ($state) {
                        'Free' => 'info',
                        'Basic' => 'success',
                        'Gold' => 'warning',
                        'Platinum' => 'gray',
                        'Sponsored' => 'danger',
                        'Influencer' => 'influencer',
                    }),

            ])
            ->filters([
                //
            ])
            ->actions([
                // Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Components\Section::make('Referrer Information')
                    // ->description('Manage account details here')
                    ->schema([
                        ImageEntry::make('profile_pic')
                            ->label('Profile Picture')
                            ->circular()
                            ->width(100)
                            ->height(100)
                            ->defaultImageUrl(asset('/img/profile-placeholder.jpg')),

                        TextEntry::make('firstname')
                            ->label('Referrer')
                            ->formatStateUsing(function ($state, $record) {
                                return $record->firstname . ' ' . $record->lastname;
                            }),

                        TextEntry::make('user_type')
                            ->label('User Type')
                            ->formatStateUsing(function ($state) {
                                $settings = app(CardDefaultsSetting::class);

                                return match ($state) {
                                    'Basic' => $settings->term_basic,
                                    'Gold' => $settings->term_gold,
                                    'Platinum' => $settings->term_platinum,
                                    'setup_incomplete' => 'Setup Incomplete',
                                    default => $state,
                                };
                            })

                    ])->columns(3)
            ]);
    }

    public static function getRelations(): array
    {
        return [
            ReferralsRelationManager::class,
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return User::query()
            ->select('users.id', 'users.firstname', 'users.lastname', 'users.email', 'users.profile_pic', 'users.user_type') // add more if needed
            ->selectRaw('COUNT(referrals.id) as referral_count')
            ->join('referrals', function ($join) {
                $join->on('users.id', '=', 'referrals.referrer_id_main')
                    ->where('referrals.referrer_type', '=', 'main');
            })
            ->groupBy('users.id', 'users.firstname', 'users.lastname', 'users.email', 'users.profile_pic', 'users.user_type');
    }

    public static function resolveRecordRouteBinding($key): ?\Illuminate\Database\Eloquent\Model
    {
        return static::getEloquentQuery()
            ->where('users.id', $key)
            ->first();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListReferrals::route('/'),
            'create' => Pages\CreateReferral::route('/create'),
            'edit' => Pages\EditReferral::route('/{record}/edit'),
            'view' => Pages\ViewReferral::route('/{record}'),
        ];
    }
}
