<?php

namespace App\Filament\Resources\ProductResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\ImageColumn;
use Filament\Forms\Components\FileUpload;
use Filament\Notifications\Notification;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;

class ProductVideoRelationManager extends RelationManager
{
    protected static string $relationship = 'media';

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return 'Videos';
    }

    protected static ?string $modelLabel = 'Video';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('path')
                    ->label('Video Link')
                    ->placeholder('Youtube or Vimeo video link')
                    ->url()
                    ->columnSpanFull()
                    ->required(),

                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->default(true)
                    ->label('Active'),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('type', 'video')->orderBy('order')
            )
            ->description("Maximum of 3 videos.")
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('path')
                    ->label('Video Link'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['product_id'] = $livewire->ownerRecord->id;
                        $data['type'] = 'video';
                        return $data;
                    })
                    ->before(function (Tables\Actions\CreateAction $action, RelationManager $livewire) {
                        $videoCount = $livewire->ownerRecord->media()->where('type', 'video')->count();

                        if ($videoCount >= 3) {
                            Notification::make()
                                ->warning()
                                ->title("Product can only add up to 3 videos!")
                                ->persistent()
                                ->send();

                            $action->halt();
                        }
                    })
                    ->disabled(function (RelationManager $livewire) {
                        $limits = [
                            'Platinum' => 5,
                            'Sponsored' => 1,
                            'Influencer' => 1,
                            'Basic' => 1,
                        ];

                        $videoCount = $livewire->ownerRecord->media()->where('type', 'video')->count();
                        return $videoCount >= 3;
                    })
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->modalHeading('Are you sure you want to delete this video?')
                    ->modalDescription('This action cannot be undone.'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->modalHeading('Are you sure you want to delete these videos?')
                        ->modalDescription('This action cannot be undone.'),
                ]),
            ]);
    }
}
