<?php

namespace App\Filament\Resources\ProductResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\ImageColumn;
use Filament\Forms\Components\FileUpload;
use Filament\Notifications\Notification;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;

class ProductImagesRelationManager extends RelationManager
{
    protected static string $relationship = 'media';

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return 'Images';
    }

    protected static ?string $modelLabel = 'Images';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                FileUpload::make('path')
                    ->directory('products')
                    ->image()
                    ->imageEditor()
                    ->columnSpanFull()
                    ->label('Upload Image')
                    ->placeholder('Upload Image')
                    ->imageCropAspectRatio('1:1')
                    ->imageResizeTargetWidth(600)
                    ->imageResizeTargetHeight(600)
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        Str::random(40) . '.' . $file->getClientOriginalExtension()
                    ),

                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->label('Active'),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('type', 'image')->orderBy('order')
            )
            ->description("Maximum of 10 images.")
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                ImageColumn::make('path')
                    ->label('Image'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                // Tables\Actions\CreateAction::make(),
                Action::make('upload-images')
                    ->label('Upload Images')
                    ->modalHeading('Upload Images')
                    ->modalButton('Upload')
                    ->form([
                        FileUpload::make('path')
                            ->reactive()
                            ->storeFiles(false)
                            ->multiple()
                            ->label('Upload Image(s)')
                            ->placeholder(function (callable $get, RelationManager $livewire) {
                                $currentCount = $livewire->ownerRecord->media()->where('type', 'image')->count();
                                $remaining = max(0, 10 - $currentCount);
                                return "Upload {$remaining} Image(s)";
                            })
                            ->directory('products')
                            ->uploadingMessage('Uploading. Please wait...')
                            ->image()
                            ->imageEditor()
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('1:1')
                            ->imageResizeTargetWidth(600)
                            ->imageResizeTargetHeight(600)
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                Str::random(40) . '.' . $file->getClientOriginalExtension()
                            )
                            ->imageEditorAspectRatios(['1:1']),
                    ])
                    // ->action(function (Action $action, array $data, RelationManager $livewire) {
                    //     $images = is_array($data['path']) ? $data['path'] : [$data['path']];
                    //     foreach ($images as $image) {
                    //         $livewire->ownerRecord->media()->create([
                    //             'path' => $image,
                    //             'type' => 'image',
                    //             'is_active' => true,
                    //             'product_id' => $livewire->ownerRecord->product_id,
                    //         ]);
                    //     }

                    //     Notification::make()
                    //         ->success()
                    //         ->title('Image(s) uploaded successfully.')
                    //         ->send();
                    // })

                    ->action(function (Action $action, array $data, RelationManager $livewire) {
                        $existingCount = $livewire->ownerRecord->media()->where('type', 'image')->count();
                        $files = is_array($data['path']) ? $data['path'] : [$data['path']];
                        $newCount = count($files);
                        $total = $existingCount + $newCount;

                        if ($total > 10) {
                            $allowed = max(0, 10 - $existingCount);

                            Notification::make()
                                ->warning()
                                ->title("You can only upload {$allowed} more image(s). Limit is 10 per product.")
                                ->persistent()
                                ->send();

                            $action->halt();
                            return;
                        }

                        foreach ($files as $file) {
                            // ✅ Move manually from tmp to intended directory
                            $storedPath = $file->store('products', 'public');

                            $livewire->ownerRecord->media()->create([
                                'path' => $storedPath,
                                'type' => 'image',
                                'is_active' => true,
                                'product_id' => $livewire->ownerRecord->product_id,
                            ]);
                        }

                        Notification::make()
                            ->success()
                            ->title('Image(s) uploaded successfully.')
                            ->send();
                    })


                    // ✅ disable modal "Upload" button while uploading
                    ->extraAttributes([
                        'wire:loading.attr' => 'disabled',
                        'wire:target' => 'data.path',
                    ])
                    ->disabled(
                        fn(RelationManager $livewire) =>
                        $livewire->ownerRecord->media()->where('type', 'image')->count() >= 10
                    ),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->modalHeading('Are you sure you want to delete this image?')
                    ->modalDescription('This action cannot be undone.'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->modalHeading('Are you sure you want to delete these images?')
                        ->modalDescription('This action cannot be undone.'),
                ]),
            ]);
    }
}
