<?php

namespace App\Filament\Resources\ProductResource\Pages;

use App\Filament\Resources\ProductResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Filament\Notifications\Notification;
use Illuminate\Support\Str;

class EditProduct extends EditRecord
{
    protected static string $resource = ProductResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('view_card')
                ->label('View')
                ->icon('heroicon-o-eye')
                // ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->user->username . '/sales/' . $record->id)
                ->url(function ($record) {
                    $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                    // if ($record->page_id && $record->subcard) {
                    //     // Sale belongs to a subcard → use parent user’s username
                    //     return $baseUrl . $record->subcard->username . '/products/' . $record->id;
                    // }

                    // if ($record->user) {
                    //     // Sale belongs directly to a user
                    //     return $baseUrl . $record->user->username . '/products/' . $record->id;
                    // }

                    if ($record->type != "subscription") {
                        if ($record->page_id && $record->subcard) {
                            return $baseUrl . $record->subcard->username . '/products/' . $record->id;
                        }

                        if ($record->user) {
                            return $baseUrl . $record->user->username . '/products/' . $record->id;
                        }
                    } else {
                        if ($record->page_id && $record->subcard) {
                            return $baseUrl . $record->subcard->username . '/subscription/' . $record->id;
                        }

                        if ($record->user) {
                            return $baseUrl . $record->user->username . '/subscription/' . $record->id;
                        }
                    }

                    return '#'; // fallback
                })
                ->openUrlInNewTab()
                ->color('info'),

            Actions\Action::make('customAction')
                ->icon('heroicon-o-document-duplicate')
                ->label('Duplicate')
                ->color('warning')
                ->requiresConfirmation()
                ->action(function (Model $record) {
                    DB::transaction(function () use ($record) {

                        // Helper function to safely duplicate a file
                        $duplicateFile = function (?string $filePath) {
                            if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                                return null;
                            }

                            $directory = dirname($filePath);
                            $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                            $originalName = pathinfo($filePath, PATHINFO_FILENAME);

                            $maxBaseLength = 50;
                            $shortName = Str::limit($originalName, $maxBaseLength, '');
                            $newFileName = $shortName . '_' . Str::random(6) . '.' . $extension;
                            $newPath = $directory . '/' . $newFileName;

                            Storage::disk('public')->makeDirectory($directory);
                            Storage::disk('public')->copy($filePath, $newPath);

                            return $newPath;
                        };

                        // ✅ Duplicate the main product
                        $newProduct = $record->replicate();
                        $newProduct->product_name = $record->product_name . ' (Copy)';
                        $newProduct->featured_image = $duplicateFile($record->featured_image);
                        $newProduct->created_at = now();
                        $newProduct->updated_at = now();
                        $newProduct->save();

                        // ✅ Duplicate related media (if any)
                        if (method_exists($record, 'media') && $record->media()->exists()) {
                            foreach ($record->media()->orderBy('order')->get() as $media) {
                                $newMedia = $media->replicate();

                                // Duplicate image files only
                                if ($media->type === 'image') {
                                    $newPath = $duplicateFile($media->path);
                                    if (!$newPath) {
                                        continue;
                                    }
                                    $newMedia->path = $newPath;
                                } else {
                                    $newMedia->path = $media->path; // reuse path for non-image
                                }

                                $newMedia->product_id = $newProduct->id;
                                $newMedia->order = $media->order;
                                $newMedia->save();
                            }
                        }

                        Notification::make()
                            ->title('Success')
                            ->body('Product duplicated successfully.')
                            ->success()
                            ->send();
                    });
                }),


            Actions\DeleteAction::make(),
        ];
    }
}
