<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OrderRecordResource\Pages;
use App\Models\OrderRecord;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Carbon\Carbon;
use Illuminate\Support\HtmlString;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\ImageEntry;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components;
use App\Models\Product;
use Illuminate\Support\Facades\Log;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Builder;

class OrderRecordResource extends Resource
{
    protected static ?string $model = OrderRecord::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Client Management';
    protected static ?string $modelLabel = 'Orders';
    protected static ?int $navigationSort = 7;

    public static function form(Form $form): Form
    {
        return $form->schema([]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->orderBy('created_at', 'desc')
            )
            ->columns([

                Tables\Columns\TextColumn::make('firstname')
                    ->label('Full Name')
                    ->formatStateUsing(fn($state, $record) => $record->firstname . ' ' . $record->lastname),

                Tables\Columns\TextColumn::make('price')
                    ->label('Price'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Ordered')
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([

                // ---------------- Order Details ----------------
                Components\Section::make('Order Details')
                    ->collapsible()
                    ->schema([
                        Components\Group::make([
                            // Receipt image
                            Components\Group::make([
                                ImageEntry::make('receipt')
                                    ->label('')
                                    ->width('100%')
                                    ->height('100%')
                                    ->defaultImageUrl(asset('/img/profile-placeholder.jpg'))
                                    ->square(),
                            ]),

                            // Order info
                            Components\Group::make([
                                TextEntry::make('price')
                                    ->label(new HtmlString('<strong>Total</strong>'))
                                    ->badge(),

                                TextEntry::make('created_at')
                                    ->label(new HtmlString('<strong>Ordered</strong>'))
                                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),

                                TextEntry::make('firstname')
                                    ->label(new HtmlString('<strong>Full Name</strong>'))
                                    ->formatStateUsing(fn($state, $record) => $record->firstname . ' ' . $record->lastname),

                                TextEntry::make('email')
                                    ->label(new HtmlString('<strong>Email</strong>')),

                                TextEntry::make('phone')
                                    ->label(new HtmlString('<strong>Phone</strong>')),

                                TextEntry::make('shipping_address')
                                    ->label(new HtmlString('<strong>Shipping Address</strong>')),

                                TextEntry::make('special_note')
                                    ->label(new HtmlString('<strong>Special Note</strong>')),
                            ]),
                        ])->columns(2),
                    ]),

                // ---------------- Products List ----------------
                Components\Section::make('Purchased Items')
                    ->collapsible()
                    ->schema([
                        Components\ViewEntry::make('products_table')
                            ->view('filament.resources.order-records.products-table')
                            ->columnSpanFull(),
                    ]),
            ]);
    }


    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOrderRecords::route('/'),
            'create' => Pages\CreateOrderRecord::route('/create'),
            'edit' => Pages\EditOrderRecord::route('/{record}/edit'),
            'view' => Pages\ViewOrderRecords::route('/{record}'),
        ];
    }
}
