<?php

namespace App\Filament\Resources;

use App\Filament\Resources\LeadCaptureFormResource\Pages;
use App\Filament\Resources\LeadCaptureFormResource\RelationManagers;
use App\Models\LeadCaptureForm;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Carbon\Carbon;
use Filament\Forms\Components\Select;
use App\Models\SubCard;
use App\Models\User;
use Filament\Forms\Set;
use Illuminate\Support\HtmlString;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use App\Settings\CardDefaultsSetting;

class LeadCaptureFormResource extends Resource
{
    protected static ?string $model = LeadCaptureForm::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Client Management';
    // protected static ?string $navigationGroup = 'Forms';
    protected static ?string $navigationLabel = 'Lead Capture Form';
    protected static ?string $modelLabel = 'Form';
    protected static ?int $navigationSort = 8;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                // 👇 Added hidden fields so Filament will save them
                Forms\Components\Hidden::make('user_id'),
                Forms\Components\Hidden::make('page_id'),

                Select::make('owner')
                    ->label('Owner')
                    ->disabled(fn() => request()->has('ownerKey'))
                    ->placeholder('Search Client or SubCard')
                    ->getSearchResultsUsing(function (string $search): array {
                        // --- Users ---
                        $users = User::query()
                            ->where(function ($query) use ($search) {
                                if (strpos($search, '@') === 0 && strlen($search) > 1) {
                                    $query->where('username', 'like', "%" . substr($search, 1) . "%");
                                } else {
                                    $query->where('firstname', 'like', "%{$search}%")
                                        ->orWhere('lastname', 'like', "%{$search}%")
                                        ->orWhere('username', 'like', "%{$search}%");
                                }
                            })
                            ->whereIn('user_type', ['Platinum', 'Gold'])
                            ->limit(25)
                            ->get()
                            ->mapWithKeys(fn($user) => [
                                "user-{$user->id}" => "{$user->firstname} {$user->lastname} - @$user->username (User)"
                            ]);

                        // --- SubCards ---
                        $subCards = SubCard::query()
                            ->where(function ($query) use ($search) {
                                $query->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            })
                            ->limit(25)
                            ->get()
                            ->mapWithKeys(fn($sub) => [
                                "sub-{$sub->id}" => "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)"
                            ]);

                        return $users->toArray() + $subCards->toArray();
                    })
                    ->getOptionLabelUsing(function ($value) {
                        if (!$value) return '';

                        [$type, $id] = explode('-', $value);

                        if ($type === 'user') {
                            $user = User::find($id);
                            return $user ? "{$user->firstname} {$user->lastname} - @$user->username (User)" : '';
                        }

                        if ($type === 'sub') {
                            $sub = SubCard::find($id);
                            return $sub ? "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)" : '';
                        }

                        return '';
                    })
                    ->formatStateUsing(function ($state, $record) {
                        if (!$record) return null;

                        if ($record->page_id && $record->subcard) {
                            return 'sub-' . $record->subcard->id;
                        }

                        if ($record->user_id && $record->user) {
                            return 'user-' . $record->user->id;
                        }


                        return null;
                    })
                    ->afterStateHydrated(function (Set $set, $state, $record) {
                        if (!$record) return;

                        if ($record->page_id && $record->subcard) {
                            $set('owner', 'sub-' . $record->subcard->id);
                        } elseif ($record->user_id && $record->user) {
                            $set('owner', 'user-' . $record->user->id);
                        }
                    })
                    ->afterStateUpdated(function (Set $set, $state) {
                        if (!$state) return;

                        [$type, $id] = explode('-', $state);

                        if ($type === 'user') {
                            // 👇 set hidden fields instead of saving directly
                            $set('user_id', $id);
                            $set('page_id', null);
                        } elseif ($type === 'sub') {
                            $set('page_id', $id);
                            $set('user_id', null);
                        }
                    })
                    ->searchable()
                    ->required()
                    ->columnSpanFull()
                    ->helperText(new HtmlString('Only <strong>Gold</strong> & <strong>Platinum</strong> users can add sales page(s). SubCards are also allowed.')),

                Forms\Components\TextInput::make('title')
                    ->label('Form Title')
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(255),

                Forms\Components\Textarea::make('description')
                    ->label('Description (Optional)')
                    ->maxLength(500)
                    ->columnSpanFull()
                    ->helperText('Provide a short overview of the form’s purpose. This text will guide users before they submit. Limit: 500 characters.'),

                FileUpload::make('image')
                    ->label('Featured Image (Optional)')
                    ->placeholder('Upload Featured Image')
                    ->directory('images')
                    ->uploadingMessage('Uploading Featured Image. Please wait...')
                    ->image()
                    ->imageEditor()
                    ->columnSpanFull()
                    ->imageResizeMode('cover')
                    ->imageCropAspectRatio(null)
                    // ->imageCropAspectRatio('1:1')
                    // ->imageResizeTargetWidth('520')
                    // ->imageResizeTargetHeight('520')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    )
                    ->imageEditorAspectRatios([
                        '1:1',
                    ]),

                Forms\Components\Toggle::make('active_image')
                    ->label('Image Visible on Page')
                    ->default(true)
                    ->columnSpanFull(),

                Forms\Components\TextInput::make('video')
                    ->label('Video Link (Optional)')
                    ->url()
                    // ->required()
                    ->columnSpanFull()
                    ->maxLength(255),

                Forms\Components\Toggle::make('active_video')
                    ->label('Video Visible on Page')
                    ->default(true)
                    ->columnSpanFull(),

                Forms\Components\TextInput::make('sender_email')
                    ->label('Sender Email')
                    ->required()
                    ->columnSpanFull()
                    ->email(),

                Forms\Components\RichEditor::make('email_received')
                    ->label('Email received by the user')
                    ->required()
                    ->maxLength(1000)
                    ->columnSpanFull()
                    ->disableGrammarly()
                    ->toolbarButtons([
                        'bold',
                        'bulletList',
                        'italic',
                        'orderedList',
                        'underline',
                    ])
                    ->default('We received your submission. Please click the link below to join the zoom call.')
                    ->helperText('This email content is what the user will receive. Max character 1000.'),


                Forms\Components\TextInput::make('link_attached')
                    ->label('Link Attached')
                    ->url()
                    ->columnSpanFull()
                    ->maxLength(255),

                Forms\Components\Toggle::make('is_active')
                    ->label('Form Active')
                    ->default(true)
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->orderBy('created_at', 'desc')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title')
                    ->label('Title'),

                Tables\Columns\TextColumn::make('user.firstname')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        if (!$record) return 'N/A';

                        if ($record->subcard) {
                            $sub = $record->subcard;
                            return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                        }

                        if ($record->user) {
                            $user = $record->user;
                            return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                        }

                        return 'N/A';
                    }),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Date')
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    // Tables\Actions\ViewAction::make(),
                    Tables\Actions\Action::make('view_card')
                        ->label('View')
                        ->icon('heroicon-o-eye')
                        // ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->user->username . '/form/' . $record->slug)
                        ->url(function ($record) {
                            if (!$record) return '#';

                            $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                            if ($record->page_id && $record->subcard) {
                                return $baseUrl . $record->subcard->username . '/form/' . $record->slug;
                            }

                            if ($record->user) {
                                return $baseUrl . $record->user->username . '/form/' . $record->slug;
                            }

                            return '#';
                        })
                        ->openUrlInNewTab()
                        ->color('info'),

                    Tables\Actions\EditAction::make(),

                    Tables\Actions\DeleteAction::make()
                        ->modalHeading('Are you sure you want to delete this form?')
                        ->modalDescription('Deleting this form will also delete all the related form submissions.This action cannot be undone.'),
                ])
                    // ->dropdown(false)
                    ->label('Actions') // optional — can be hidden if you prefer
                    ->icon('heroicon-o-cog-6-tooth') // optional
                    ->color('gray'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLeadCaptureForms::route('/'),
            'create' => Pages\CreateLeadCaptureForm::route('/create'),
            'edit' => Pages\EditLeadCaptureForm::route('/{record}/edit'),
        ];
    }
}
