<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ButtonResource\Pages;
use App\Filament\Resources\ButtonResource\RelationManagers;
use App\Models\Button;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Filament\Support\Enums\IconPosition;

class ButtonResource extends Resource
{
    protected static ?string $model = Button::class;

    protected static ?string $navigationIcon = 'heroicon-o-link';

    protected static ?string $navigationGroup = 'Client Management';

    protected static ?string $navigationLabel = 'Links';

    protected static ?string $modelLabel = 'Link';

    protected static ?string $slug = 'links';

    protected static ?int $navigationSort = 2;

    // hide this resource
    public static function shouldRegisterNavigation(): bool
    {
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('user_id')
                    ->label('Owner')
                    ->getSearchResultsUsing(
                        fn(string $search): array =>
                        User::where(function ($query) use ($search) {
                            $query->where('firstname', 'like', "%{$search}%")
                                ->orWhere('lastname', 'like', "%{$search}%");
                        })
                            ->limit(50)
                            ->pluck(DB::raw('CONCAT(firstname, " ", lastname)'), 'id')
                            ->toArray()
                    )
                    ->getOptionLabelUsing(
                        fn($value) =>
                        optional(User::find($value))->firstname . ' ' . optional(User::find($value))->lastname
                    )
                    ->searchable()
                    ->required()
                    ->afterStateUpdated(function (Set $set, $state) {
                        // Check if the selected user has 5 or more videos
                        $user = User::find($state);
                        if ($user->user_type == "Free" && $user->buttons()->count() >= 4) {
                            // Trigger a notification
                            Notification::make()
                                ->title('Limit reached')
                                ->body('Free card holders can only add up to 4 links!')
                                ->warning()
                                ->send();
                            // Optionally, you can reset the state or show a validation error:
                            $set('user_id', null); // Reset the user_id if limit is reached
                        }
                    }),
                Forms\Components\TextInput::make('type')
                    ->hidden()
                    ->default('link'),
                // Forms\Components\Select::make('type')
                //     ->required()
                //     ->label('Button Type')
                //     ->options([
                //         'link' => 'Link',
                //         'email' => 'Email',
                //         'phone' => 'Phone',
                //         'sms' => 'SMS',
                //         'whatsapp' => 'WhatAapp',
                //         'zoom' => 'Zoom',
                //         'cashapp' => 'CashApp',
                //         'paypal' => 'Paypal',
                //     ]),
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('link')
                    ->required()
                    ->maxLength(255),
                // Forms\Components\TextInput::make('type')
                //     ->required(),
                // Forms\Components\TextInput::make('order')
                //     ->numeric()
                //     ->default(null),
                // Forms\Components\TextInput::make('user_id')
                //     ->required()
                //     ->numeric(),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                Tables\Columns\TextColumn::make('user.firstname')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        return $record->user ? $record->user->firstname . ' ' . $record->user->lastname : 'N/A';
                    })
                    ->sortable()
                    ->searchable(['firstname', 'lastname']),
                // Tables\Columns\TextColumn::make('user.lastname')
                //     ->searchable()
                //     ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    // ->url(fn($record) => $record->type === 'email' ? 'mailto:' . $record->link : $record->link)
                    ->url(
                        fn($record) =>
                        $record->type === 'email' ? 'mailto:' . $record->link : ($record->type === 'sms' ? 'sms:' . $record->link : ($record->type === 'phone' ? 'tel:' . $record->link : $record->link))
                    )
                    ->openUrlInNewTab(),
                // Tables\Columns\TextColumn::make('type')
                //     ->searchable()
                //     ->formatStateUsing(fn($state) => ucfirst($state)),
                // Tables\Columns\TextColumn::make('link')
                //     ->searchable(),
                // Tables\Columns\TextColumn::make('type'),
                // Tables\Columns\TextColumn::make('order')
                //     ->numeric()
                //     ->sortable(),
                // Combine the first name and last name in one column

                // Tables\Columns\IconColumn::make('is_active')
                //     ->boolean(),
                // Tables\Columns\TextColumn::make('created_at')
                //     ->dateTime()
                //     ->sortable()
                //     ->toggleable(isToggledHiddenByDefault: true),
                // Tables\Columns\TextColumn::make('updated_at')
                //     ->dateTime()
                //     ->sortable()
                //     ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
                // SelectFilter::make('type')
                //     ->label('Button Type')
                //     ->options([
                //         'link' => 'Link',
                //         'email' => 'Email',
                //         'phone' => 'Phone',
                //         'sms' => 'SMS',
                //         'whatsapp' => 'WhatAapp',
                //         'zoom' => 'Zoom',
                //         'cashapp' => 'CashApp',
                //         'paypal' => 'Paypal',
                //     ]),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                // Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        // auth()->user()
        return parent::getEloquentQuery()->with('user');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListButtons::route('/'),
            'create' => Pages\CreateButton::route('/create'),
            'view' => Pages\ViewButton::route('/{record}'),
            'edit' => Pages\EditButton::route('/{record}/edit'),
        ];
    }
}
